//
//  BDClockItem.swift
//  CTFit
//
//  Created by Mac on 2021/11/3.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation

enum BDWeekType: Int, BDEnumtProtocol {
    case sun
    case mon
    case tue
    case wed
    case thu
    case fri
    case sat

    var value: UInt8  {
        switch self {
        case .sun: return 0x01
        case .mon: return 0x02
        case .tue: return 0x04
        case .wed: return 0x08
        case .thu: return 0x10
        case .fri: return 0x20
        case .sat: return 0x40
        }
    }
    
    static func value(_ repeats:[BDWeekType]) -> UInt8 {
        var value: UInt8 = 0
        repeats.forEach { (e) in value |= e.value }
        return value
    }
    
    static func parse(_ value: UInt8) -> [BDWeekType] {
        var repeats: [BDWeekType] = []
        collection.forEach { (e) in if e.value & value == e.value { repeats.append(e)} }
        return repeats
    }
    
    static var none: UInt8     = 0x00
    static var workday: UInt8  = 0x3E
    static var everyday: UInt8 = 0x7F
    static var collection: [BDWeekType] = [.sun, .mon, .tue, .wed, .thu, .fri, .sat]
}

extension BDWeekType: Equatable {
    static func == (lhs: BDWeekType, rhs: BDWeekType) -> Bool {
        lhs.value == rhs.value
    }
}

extension Array where Element == BDWeekType {
    static func == (lhs: [Element], rhs: [Element]) -> Bool {
        Element.value(lhs) == Element.value(rhs)
    }
}


///: - Initialize
class BDClockItem: BDModelProtocol {
    required init() {
        let date = Date()
        let calendar = CalendarUtils.calendar
        
        self.enable = true
        self.repeats = [BDWeekType]()
        self.hour = UInt8(calendar.component(.hour, from: date))
        self.minute = UInt8(calendar.component(.minute, from: date))
        self.vibrate = BDVibrate()
        self.label = ""
    }
    
    /// 开关
    var enable: Bool
    /// 小时 ( 0 ~ 23 )
    var hour: UInt8
    /// 分钟 ( 0 ~ 59 )
    var minute: UInt8
    /// 周重复
    var repeats: [BDWeekType]
    /// 震动
    var vibrate: BDVibrate
    /// 闹钟标签：max ( 15 Byte )
    var label: String
    
    var date: Date {
        get {
            CalendarUtils.date(withHour: Int(hour), minute: Int(minute))
        }
        set {
            let components = CalendarUtils.components(ofDate: newValue)
            hour = UInt8(components.hour!)
            minute = UInt8(components.minute!)
        }
    }
    
    var time: String { return String(format: "%02d:%02d", hour, minute) }
}

extension BDClockItem: Equatable {
    static func == (lhs: BDClockItem, rhs: BDClockItem) -> Bool {
        guard lhs.enable == rhs.enable,
              lhs.hour == rhs.hour, lhs.minute == rhs.minute,
              lhs.repeats == rhs.repeats, lhs.label == rhs.label else { return false }
        return true
    }
}


import BandKit
extension BDWeekType {
    var band_z: SRWeekType {
        switch self {
        case .sun: return .sun
        case .mon: return .mon
        case .tue: return .tue
        case .wed: return .wed
        case .thu: return .thu
        case .fri: return .fri
        case .sat: return .sat
        }
    }
    static func parse(_ t: SRWeekType) -> BDWeekType {
        switch t {
        case .sun: return .sun
        case .mon: return .mon
        case .tue: return .tue
        case .wed: return .wed
        case .thu: return .thu
        case .fri: return .fri
        case .sat: return .sat
        }
    }
}

extension BDClockItem {
    var band_z: SRClockItem {
        let m = SRClockItem()
        m.enable = enable
        m.hour = hour
        m.minute = minute
        m.repeats = repeats.map({ $0.band_z })
        m.vibrate = vibrate.band_z
        m.label = label
        return m
    }
    static func parse(_ t: SRClockItem) -> BDClockItem {
        let m = BDClockItem()
        m.enable = t.enable
        m.hour = t.hour
        m.minute = t.minute
        m.repeats = t.repeats.map({ BDWeekType.parse($0) })
        m.vibrate.update(t.vibrate)
        m.label = t.label
        return m
    }
}


import YFitKit
extension BDClockItem {
    var band_s: YFClockItem {
        let m = YFClockItem()
        m.status = (enable ? 1 : 0)
        m.hour = hour
        m.min = minute
        m.repeatDay = BDWeekType.value(repeats) | 0x80
        m.name = label
        return m
    }
    static func parse(_ t: YFClockItem) -> BDClockItem {
        let m = BDClockItem()
        m.enable = (t.status > 0)
        m.hour = t.hour
        m.minute = t.min
        m.repeats = BDWeekType.parse(t.repeatDay)
        m.label = t.name
        return m
    }
}



