//
//  BandRecords.swift
//  CTFit
//
//  Created by Mac on 2021/10/13.
//  Copyright © 2021 jpaxm. All rights reserved.
//

import Foundation
import BandKit
import YFitKit

// MARK: - 数据查询
extension BandData {
    ///: - home
    static func queryHome(with date: Date, series_s more:Bool = false, callback: @escaping (BDTotal?)->Void) {
        DispatchQueue.global().async {
            let m = BDTotal.query(with: date, series_s: more)
            DispatchQueue.main.async { callback(m) }
        }
    }
    
    ///: - steps for detail
    static func queryStepDetail(with date: Date, callback: @escaping (BDTotal?, BDStep?, BDExercise?)->Void) {
        DispatchQueue.global().async {
            if BandConnect.protocolType  == .series_z {
                let tuples = BandRecord.queryStepDetail(with: date)
                var _total: BDTotal? = nil
                var _step: BDStep? = nil
                var _exercise: BDExercise? = nil
                if let m = tuples.0 { _total    = BDTotal.parse(m) }
                if let m = tuples.1 { _step     = BDStep.parse(m) }
                if let m = tuples.2 { _exercise = BDExercise.parse(m) }
                DispatchQueue.main.async { callback(_total, _step, _exercise) }
            }
            if BandConnect.protocolType  == .series_s {
                let total = BDTotal.query(with: date)
                let step = BDStep.query(with: date)
                let exercise = BDExercise.query(with: date, series_s: true)
                DispatchQueue.main.async { callback(total, step, exercise) }
            }
        }
    }
    
    ///: - steps for interval
    static func queryStepInterval(frome start: Date, to end: Date, callback: @escaping ([BDTotal], [BDExercise])->Void) {
        DispatchQueue.global().async {
            if BandConnect.protocolType  == .series_z {
                let tuples = BandRecord.queryStepInterval(frome: start, to: end)
                DispatchQueue.main.async { callback(tuples.0.map({ BDTotal.parse($0) }), tuples.1.map({ BDExercise.parse($0) })) }
            }
            
            if BandConnect.protocolType  == .series_s {
                let totalArray = YFTotal.query(from: start, to: end)
                let exerciseArray = YFExercise.query(from: start, to: end)
                DispatchQueue.main.async { callback(totalArray.map({ BDTotal.parse($0) }), exerciseArray.map({ BDExercise.parse($0, false) })) }
            }
        }
    }
    
    ///: - heart
    static func queryHeart(with date: Date, callback: @escaping (BDHeart?)->Void) {
        DispatchQueue.global().async {
            let m = BDHeart.query(details: date)
            DispatchQueue.main.async { callback(m) }
        }
    }
    
    ///: - sleep for detail
    static func querySleepDetail(with date: Date, callback: @escaping (BDSleep?, BDHeart?) -> Void) {
        DispatchQueue.global().async {
            let sleep = BDSleep.query(with: date, series_s: true)
            let heart = BDHeart.query(sleep: date)
            DispatchQueue.main.async { callback(sleep, heart) }
        }
    }
    
    ///: - sleep for interval
    static func querySleepInterval(frome start: Date, to end: Date, callback: @escaping ([BDSleep]) -> Void) {
        DispatchQueue.global().async {
            let array = BDSleep.query(from: start, to: end)
            DispatchQueue.main.async { callback(array) }
        }
    }
    
    ///: - Detect info item  ( limit: 300 )
    static func queryDetect(callback: @escaping () -> Void) {
        DispatchQueue.global().async {
            BDDetect.read()
            DispatchQueue.main.async { callback() }
        }
    }
    
}

// MARK: - 记录数据模型
///: - 计步详细
struct BDStep {
    /// 日期
    var dateStr: String = ""
    /// 总步数
    var total: UInt32 = 0
    /// 每小时的计步数据记录 ( 24 小时 )
    var hourRecords: [UInt16] = [UInt16](repeating: 0, count: 24)
    
    fileprivate static func parse(_ m: SRStep) -> BDStep {
        return BDStep(dateStr: m.date.dateStr, total: m.total, hourRecords: m.hourRecords)
    }

    fileprivate static func parse(_ m: YFTotalDay) -> BDStep {
        let records = m.hourIntervalItems.map({ UInt16(truncatingIfNeeded: $0.walkSteps) })
        return BDStep(dateStr: m.dateStr, total: m.latest?.walkSteps ?? 0, hourRecords: records)
    }
}

///: - 总数据
struct BDTotal {
    /// 日期
    var dateStr: String = ""
    /// 步数
    var steps: UInt32 = 0
    /// 卡路里
    var calories: Double = 0
    /// 距离 （ m）
    var distance: Double = 0
    /// 心率
    var latestHeart: UInt8 = 0
    /// 运动时间（分钟）
    var exerciseMinutes: UInt16 = 0
    /// 睡眠时间（分钟）
    var sleepMinutes: UInt16 = 0
    
    fileprivate static func parse(_ m: SRTotal) -> BDTotal {
        return BDTotal(dateStr: m.date.dateStr, steps: m.steps, calories: Double(m.calories) * 1000, distance: Double(m.distance) * 10, latestHeart: m.latestHeart, exerciseMinutes: m.exerciseMinutes, sleepMinutes: m.sleepMinutes)
    }
    
    fileprivate static func parse(_ m: YFTotal) -> BDTotal {
        return BDTotal(dateStr: m.dateStr, steps: m.walkSteps, calories: Double(m.walkCalories), distance: Double(m.walkDistance))
    }
    
    fileprivate static func parse(_ m: YFTotal, latestHeart: UInt8, exerciseMinutes: UInt16, sleepMinutes: UInt16) -> BDTotal {
        return BDTotal(dateStr: m.dateStr, steps: m.walkSteps, calories: Double(m.walkCalories), distance: Double(m.walkDistance),
                       latestHeart: latestHeart, exerciseMinutes: exerciseMinutes, sleepMinutes: sleepMinutes)
    }
}

///: -  心率
struct BDHeart {
    /// 日期
    var dateStr: String = ""
    /// 最高
    var highest: UInt8 = 0
    /// 最底
    var lowest: UInt8 = 0
    /// 平均
    var aver: UInt8 = 0
    /// 最近的
    var latest: UInt8 = 0
    
    /// 一天的静态心率数据  ( 每10分钟记录一次，一天144个 ),  可以是随眠数据（跨天， 96个）
    var records = [UInt8]()
    
    fileprivate static func parse(_ m: SRHeart) -> BDHeart {
        return BDHeart(dateStr: m.date.dateStr, highest: m.highest, lowest: m.lowest, aver: m.aver, latest: m.lowest, records: m.records)
    }
    
    fileprivate static func parse(_ m: YFHeartDay) -> BDHeart {
        return BDHeart(dateStr: m.dateStr, highest: m.highest, lowest: m.lowest, aver: m.average, latest: m.lowest, records: m.values)
    }
}

///: -  睡眠
struct BDSleep {
    /// 日期
    var dateStr: String = ""
    /// Period of time
    var periodTime: String = "--:--"
    /// 总时间长 （ 分钟 ）
    var totalMinutes: UInt16 = 0
    /// 深睡时长（ 分钟 ）
    var deeplMinutes: UInt16 = 0
    /// 浅睡时长（ 分钟 ）
    var lightMinutes: UInt16 = 0
    /// 清醒时长（ 分钟 ）
    var wakeMinutes: UInt16 = 0

    /// 一天的睡眠数据  ( 昨天 20:00 ~ 今天 12:00,  每分钟记录一次，总共 960 次 )
    var records = [BDSleepType]()

    fileprivate static func parse(_ m: SRSleep) -> BDSleep {
        return BDSleep(dateStr: m.date.dateStr, periodTime: m.periodTime, totalMinutes: m.totalMinutes, deeplMinutes: m.deeplMinutes, lightMinutes: m.lightMinutes, wakeMinutes: m.wakeMinutes, records: m.records.map({ BDSleepType.parse($0) }))
    }
    
    fileprivate static func parse(_ m: YFSleepDay) -> BDSleep {
        return BDSleep(dateStr: m.dateStr, periodTime: m.periodTime, totalMinutes: m.totalMinutes, deeplMinutes: m.deeplMinutes, lightMinutes: m.lightMinutes, wakeMinutes: m.wakeMinutes, records: m.records.map({ BDSleepType.parse($0) }))    }
}

///: -  训练
struct BDExercise {
    // 日期
    var dateStr: String = ""
    /// 总运动分钟数
    var totalMinutes: Int = 0
    /// 一天的运动锻炼数据
    var records = [BDExerciseItem]()
   
    fileprivate static func parse(_ m: SRExercise) -> BDExercise {
        return BDExercise(dateStr: m.date.dateStr, totalMinutes: m.totalMinutes, records: m.records.map({ BDExerciseItem.parse($0)}))
    }
    fileprivate static func parse(_ m: YFExercise, _ heart: Bool) -> BDExercise {
        return BDExercise(dateStr: m.dateStr, totalMinutes: m.totalMinutes, records: m.records.map({ BDExerciseItem.parse($0, heart)}))
    }
}

struct BDExerciseItem {
    
    /// 运动类型
    var type: BDSportType = .normal
    /// 持续时间 ( 分钟 )
    var duration: UInt16 = 0
    /// 开始小时 （0~23）
    var startHour: UInt8  = 0
    /// 开始分钟 （0~59）
    var startMinute: UInt8  = 0
    /// 结束小时 （0~23）
    var endHour: UInt8  = 0
    /// 结束分钟 （0~59）
    var endMinute: UInt8  = 0
    
    /// 步数
    var steps: UInt32  = 0
    /// 卡路里 （大卡）
    var calories: UInt16  = 0
    /// 距离 （10 m）
    var distance: UInt16  = 0
    
    /// 最大步频 ( 每分钟的步数 )
    var maxSpace: UInt8  = 0
    /// 平均步频 ( 每分钟的步数 )
    var avgSpace: UInt8  = 0
    
    /// 最大心率
    var maxHeart: UInt8  = 0
    /// 平均心率
    var avgHeart: UInt8  = 0
    /// 运动心率记录 （每分钟一次）
    var heartRecords = [UInt8]()
    
    /// Period of time
    var periodTime: String {
        String(format: "%02d:%02d~%02d:%02d", min(startHour, 23), min(startMinute, 59), min(endHour, 23), min(endMinute, 59))
    }
    
    fileprivate static func parse(_ m: SRExerciseItem) -> BDExerciseItem {
        return BDExerciseItem(type: BDSportType.parse(m.type), duration: m.duration, startHour: m.startHour, startMinute: m.startMinute, endHour: m.endHour, endMinute: m.endMinute, steps: m.steps, calories: m.calories, distance: m.distance, maxSpace: m.maxSpace, avgSpace: m.avgSpace, maxHeart: m.maxHeart, avgHeart: m.avgHeart, heartRecords: m.heartRecords)
    }
    
    fileprivate static func parse(_ m: YFExerciseItem, _ heart: Bool) -> BDExerciseItem {
        let _duration = UInt16(m.total_time/60)
        
        let _startDate   = Date(timeIntervalSince1970: m.start_time)
        let _startHour   = UInt8(CalendarUtils.hour(ofDate: _startDate))
        let _startMinute = UInt8(CalendarUtils.minute(ofDate: _startDate))
        
        let _endDate  = Date(timeIntervalSince1970: m.end_time)
        let _endHour  = UInt8(CalendarUtils.hour(ofDate: _endDate))
        let _endMinute = UInt8(CalendarUtils.minute(ofDate: _endDate))

        let _multiple = (m.max_speed==0||m.speed==0) ? 1 : m.max_speed/m.speed
        let _avgSpace = UInt8(truncating: NSNumber(value: (_duration==0) ? 0 : m.step/UInt32(_duration)))
        let _maxSpace = UInt8(truncating: NSNumber(value: Double(_avgSpace) * _multiple))
        
        var model = BDExerciseItem(type: BDSportType.parse(m.sport_type), duration: _duration, startHour: _startHour, startMinute: _startMinute, endHour: _endHour, endMinute: _endMinute, steps: m.step, calories: m.calories, distance: m.distance, maxSpace: _maxSpace, avgSpace: _avgSpace, maxHeart: m.max_hr, avgHeart: m.aver_hr)
        if heart, let exerciseHeart = YFHeartExercise.query(with: m) {
            model.heartRecords = exerciseHeart.values
        }
        return model
    }
}

///: -  检测
struct BDDetect {
    /// 类型
    var type: BDDetectType = .bpm
    /// 时间戳 (单位 秒 ) 相对于1970 的时间
    /// yyyy-MM-dd HH:mm
    var time: TimeInterval = 0
    /// 心率 或者 血压高压值 (收缩压)
    var heartorsbp: UInt8 = 0
    /// 血氧 或者 血压低压值 (舒张压)
    var spo2ordbp: UInt8 = 0
    /// 结束检测
    var end: Bool = false
    
    var timeText: String { return dformatterTime.string(from: Date(timeIntervalSince1970: time)) }
    var dateText: String { return dformatterDate.string(from: Date(timeIntervalSince1970: time)) }
    var dateTime: String { return dformatterDateTime.string(from: Date(timeIntervalSince1970: time)) }
    
    fileprivate static func parse(_ m: SRDetectItem) -> BDDetect {
        return BDDetect(type: BDDetectType.parse(m.type), time: m.time, heartorsbp: m.heartorsbp, spo2ordbp: m.spo2ordbp, end: m.end)
    }
   
    ///: - private
    private let dformatterTime: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "HH:mm"
           return dformatter
    }()
    private let dformatterDate: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "MM-dd"
           return dformatter
    }()
    private let dformatterDateTime: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "yyyy-MM-dd HH:mm"
           return dformatter
    }()
}

extension BDDetect: Equatable {
    static func ==(lhs: BDDetect, rhs: BDDetect) -> Bool {
        guard lhs.type == rhs.type,
              lhs.time == rhs.time,
              lhs.heartorsbp == rhs.heartorsbp,
              lhs.spo2ordbp == rhs.spo2ordbp,
              lhs.end == rhs.end else { return false }
        return true
    }
}

extension BDDetect {
    static var heartItems = [BDDetect]()
    static var pressureItems = [BDDetect]()
    static func reloadItems() {
        heartItems = SRDetectItem.heartRateItems.map({ BDDetect.parse($0) })
        pressureItems = SRDetectItem.bloodPressureItems.map({ BDDetect.parse($0) })
    }
    
    fileprivate static func read() {
        SRDetectItem.readFromDB()
        reloadItems()
    }
}

// MARK: - query for Extension
extension BDStep{
    fileprivate static func query(with date: Date) -> BDStep? {
        if BandConnect.protocolType  == .series_z {
            if let m = SRStep.query(with: date) { return BDStep.parse(m) }
        }
        if BandConnect.protocolType  == .series_s {
            if let m = YFTotalDay.query(with: date) { return BDStep.parse(m) }
        }
        return nil
    }
}

extension BDTotal {
    fileprivate static func query(with date: Date, series_s more: Bool = false) -> BDTotal? {
        if BandConnect.protocolType  == .series_z {
            if let m = SRTotal.query(with: date) { return BDTotal.parse(m) }
        }
        if BandConnect.protocolType  == .series_s {
            if !more {
                var total = BDTotal()
                total.dateStr = CalendarUtils.dateStr(date: date)
                if let m = YFTotal.queryLatest(with: date) {
                    total.steps = m.walkSteps
                    total.calories = Double(m.walkCalories)
                    total.distance = Double(m.walkDistance)
                }
                if let m = YFHeart.queryLatest(with: date) {
                    total.latestHeart = m.heartNum
                }
                return total
            } else {
                var total = BDTotal()
                total.dateStr = CalendarUtils.dateStr(date: date)
                if let m = YFTotal.queryLatest(with: date) {
                    total.steps = m.walkSteps
                    total.calories = Double(m.walkCalories)
                    total.distance = Double(m.walkDistance)
                }
                if let m = YFHeart.queryLatest(with: date) {
                    total.latestHeart = m.heartNum
                }
                if let m = YFSleepDay.query(with: date) {
                    total.sleepMinutes = m.totalMinutes
                }
                if let m = YFExercise.query(with: date) {
                    total.exerciseMinutes = UInt16(m.totalMinutes)
                }
                return total
            }
        }
        return nil
    }
}


extension BDHeart {
    fileprivate static func query(details date: Date) -> BDHeart? {
        if BandConnect.protocolType  == .series_z {
            if let m = SRHeart.query(details: date) { return BDHeart.parse(m) }
        }
        if BandConnect.protocolType  == .series_s {
            if let m = YFHeartDay.query(details: date) { return BDHeart.parse(m) }
        }
        return nil
    }
    
    fileprivate static func query(sleep date: Date) -> BDHeart? {
        if BandConnect.protocolType  == .series_z {
            if let m = SRHeart.query(sleep: date) { return BDHeart.parse(m) }
        }
        if BandConnect.protocolType  == .series_s {
            if let m = YFHeartDay.query(sleep: date) { return BDHeart.parse(m) }
        }
        return nil
    }
}

extension BDSleep{
    fileprivate static func query(with date: Date, series_s details: Bool = false) -> BDSleep? {
        if BandConnect.protocolType  == .series_z {
            if let m = SRSleep.query(with: date) { return BDSleep.parse(m) }
        }
        if BandConnect.protocolType  == .series_s {
            if let m = YFSleepDay.query(with: date, details: details) { return BDSleep.parse(m) }
        }
        return nil
    }
    
    fileprivate static func query(from fromDate: Date, to toDate: Date) -> [BDSleep] {
        if BandConnect.protocolType  == .series_z {
            let m = SRSleep.query(from: fromDate, to: toDate)
            return m.map({ BDSleep.parse($0) })
        }
        if BandConnect.protocolType  == .series_s {
            let m = YFSleepDay.query(from: fromDate, to: toDate)
            return m.map({ BDSleep.parse($0) })
        }
        return []
    }

}


extension BDExercise {
    fileprivate static func query(with date: Date, series_s heart: Bool = false) -> BDExercise? {
        if BandConnect.protocolType  == .series_z {
            if let m = SRExercise.query(with: date) { return BDExercise.parse(m) }
        }
        if BandConnect.protocolType  == .series_s {
            if let m = YFExercise.query(with: date) { return BDExercise.parse(m, heart) }
        }
        return nil
    }
}




