//
//  BandDataUtils.swift
//  CTFit
//
//  Created by Mac on 2021/10/13.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation
import BandKit
import YFitKit

// MARK: - 数据类型

enum BDDataType {
    
    case none
    
    /// 通知
    case notify(BDNotifyType)

    ///: - series_z
    case firmwareInfo, userInfo, clockInfo, drinkInfo
    case record(type: BDRecordType, dateStr: String?)
    case detectInfo([(type: BDDetectType, detected: Bool)])

    ///: - series_s
    case hybridInfo, otadevice
    
    static func parse(_ type: SRDataType) -> BDDataType {
        switch type {
        case .notify(let t): return .notify(BDNotifyType.parse(t))
        case .firmwareInfo: return .firmwareInfo
        case .userInfo:     return .userInfo
        case .clockInfo:    return .clockInfo
        case .drinkInfo:    return .drinkInfo
        case .detectInfo(let array):  return .detectInfo(array.map({ (BDDetectType.parse($0.type), $0.detected) }))
        case .record(let type, _, let date): return record(type: BDRecordType.parse(type), dateStr: date.dateStr)
        default: return .none
        }
    }
    
    static func parse(_ type: YFDataType) -> BDDataType {
        switch type {
        case .notify(let t): return .notify(BDNotifyType.parse(t))
        case .hybridInfo:   return .hybridInfo
        case .otadevice:    return .otadevice
        case .record(let type, let dateStr): return record(type: BDRecordType.parse(type), dateStr: dateStr)
        }
    }
}

/// 记录数据类型
enum BDRecordType {
    case total      // 总数据
    case step       // 计步
    case heart      // 心率
    case sleep      // 睡眠
    case exercise   // 训练
    
    var band_z: SRRecordType {
        switch self {
        case .total:   return .total
        case .step:    return .step
        case .heart:   return .heart
        case .sleep:    return .sleep
        case .exercise: return .exercise
        }
    }
    
    
    static func parse(_ type: SRRecordType) -> BDRecordType {
        switch type {
        case .total:   return total
        case .step:    return step
        case .heart:   return heart
        case .sleep:    return sleep
        case .exercise: return exercise
        }
    }
    
    static func parse(_ type: YFRecordType) -> BDRecordType {
        switch type {
        case .total:    return .total
        case .heart:    return .heart
        case .sleep:    return .sleep
        case .exercise: return .exercise
        }
    }
};

///: - 设备事件通知类型
enum BDNotifyType {
    /// 训练
    enum Exercise: Equatable {
        case start  // 开启
        case stop   // 停止
        case ignore // 忽略
        case sport(BDSportType)   // 运动类型
        
        var ext: UInt8 {
            switch self {
            case .start, .stop, .ignore: return 0x00
            case .sport(_):              return 0x02
            }
        }
        var byte1: UInt8 {
            switch self {
            case .start:    return 0x00
            case .stop:     return 0x01
            case .ignore:   return 0x02
            case .sport(let type): return type.value
            }
        }
        
        public var value: UInt16 {
            (UInt16(ext) << 8) + UInt16(byte1)
        }
       
        public static func parse(_ value: UInt16) ->  Exercise? {
            let ext:   UInt8 = UInt8(truncating: NSNumber(value: value>>8))
            let byte1: UInt8 = UInt8(truncating: NSNumber(value: value))
            switch ext {
            case 0x00:
                switch byte1 {
                case 0x00: return .start
                case 0x01: return .stop
                case 0x02: return .ignore
                default:  break
                }
            case 0x02:
                switch byte1 {
                case 0x00: break
                default: if let type = BDSportType.parse(byte1) { return .sport(type) };
                }
            default: break
            }
            return nil
        }
        
        public var description: String {
            switch self {
            case .start:  return "开启"
            case .stop:   return "停止"
            case .ignore: return "忽略"
            case .sport(let type): return "运动(\(type))"
            }
        }
        public static func == (lhs: Exercise, rhs: Exercise) -> Bool {
            lhs.ext == rhs.ext && lhs.byte1 == rhs.byte1
        }
        
        
        fileprivate static func parse(_ type: SRNotifyType.Exercise) -> Exercise {
            switch type {
            case .start:  return .start
            case .stop:   return .stop
            case .ignore: return .ignore
            case .sport(let t):  return .sport(BDSportType.parse(t))
            }
        }

        fileprivate static func parse(_ type: YFNotifyType.Exercise) -> Exercise {
              return .stop
        }        
    }
    
    /// 相机
    enum Camera {
        case on          // 打开
        case off         // 关闭
        case takePicture // 拍照
        
        fileprivate static func parse(_ type: SRNotifyType.Camera) -> Camera {
            switch type {
            case .enter: return .on
            case .exit:  return .off
            case .takePicture: return .takePicture
            }
        }

        fileprivate static func parse(_ type: YFNotifyType.Camera) -> Camera {
            switch type {
            case .exit:   return .off
            case .take: return .takePicture
            }
        }
        
    }
    
    /// 查找手机
    enum FindPhone {
        case on  // 打开
        case off // 关闭
        
        fileprivate static func parse(_ type: SRNotifyType.FindPhone) -> FindPhone {
            switch type {
            case .start: return .on
            case .stop:  return .off
            }
        }

        fileprivate static func parse(_ type: YFNotifyType.FindPhone) -> FindPhone {
            switch type {
            case .start: return .on
            case .stop:  return .off
            }
        }
    }
    
    case exercise(Exercise)
    case camera(Camera)
    case findPhone(FindPhone)
    
    fileprivate static func parse(_ type: SRNotifyType) -> BDNotifyType {
        switch type {
        case .exercise(let t): return .exercise(Exercise.parse(t))
        case .camera(let t): return .camera(Camera.parse(t))
        case .findPhone(let t): return .findPhone(FindPhone.parse(t))
        }
    }
    
    fileprivate static func parse(_ type: YFNotifyType) -> BDNotifyType {
        switch type {
        case .exercise(let t): return .exercise(Exercise.parse(t))
        case .camera(let t): return .camera(Camera.parse(t))
        case .findPhone(let t): return .findPhone(FindPhone.parse(t))
        }
    }
}

///: -  检测的类型
enum BDDetectType {
    case bpm, bp
    static func parse(_ type: SRDetectType) -> BDDetectType {
        switch type {
        case .bpm: return .bpm
        case .bp: return .bp
        }
    }
    
    var band_z: SRDetectType {
        switch self {
        case .bpm: return .bpm
        case .bp: return .bp
        }
    }
}

///: -  运动类型
enum BDSportType {
    case normal     // 普通计步模式
    case run        // 跑步
    case climbing   // 爬山
    case indoor     // 室内运动
    case swiming    // 游泳
    case yoga       // 瑜伽

    case walk       // 健步
    case badminton  // 羽毛球
    case basketball // 篮球
    case football   // 足球
    case pingPong   // 乒乓球
    case tennis     // 网球
    case bike       // 骑行
    case indoorRun  // 室内跑
   
    var value: UInt8 {
        switch self {
        case .normal:       return 0x01
        case .run:          return 0x07
        case .climbing:     return 0x09
        case .indoor:       return 0x0A
        case .swiming:      return 0x0B
        case .yoga:         return 0x0C

        case .walk:         return 0x77
        case .badminton:    return 0x80
        case .basketball:   return 0x81
        case .football:     return 0x82
        case .pingPong:     return 0x85
        case .tennis:       return 0x87
        case .bike:         return 0x88
        case .indoorRun:    return 0x8A
        }
    }
    
    static var collection: [BDSportType] = [.normal, .run, .climbing, .indoor, .swiming, .yoga, .walk,
                                          .badminton, .basketball, .football, .pingPong, .tennis, bike, .indoorRun]

    static func parse(_ value: UInt8) -> BDSportType? {
        var type: BDSportType? = nil
        for e in collection { if e.value==value { type = e; break }}
        return type
    }
    
    var text: String {
        switch self {
        case .normal:       return SRString.Exercise.walk.locastr
        case .run:          return SRString.Exercise.run.locastr
        case .climbing:     return SRString.Exercise.climbing.locastr
        case .indoor:       return SRString.Exercise.indoor.locastr
        case .swiming:      return SRString.Exercise.swiming.locastr
        case .yoga:         return SRString.Exercise.yoga.locastr
        
        case .walk:         return SRString.Exercise.walk.locastr
        case .badminton:    return SRString.Exercise.badminton.locastr
        case .basketball:   return SRString.Exercise.basketball.locastr
        case .football:     return SRString.Exercise.football.locastr
        case .pingPong:     return SRString.Exercise.pingpong.locastr
        case .tennis:       return SRString.Exercise.tennis.locastr
        case .bike:         return SRString.Exercise.bike.locastr
        case .indoorRun:    return SRString.Exercise.indoor_run.locastr
        }
    }
    
    var image: UIImage {
        switch self {
        case .normal:       return ImageRepo.SportTypes.normal
        case .run:          return ImageRepo.SportTypes.run
        case .climbing:     return ImageRepo.SportTypes.climbing
        case .indoor:       return ImageRepo.SportTypes.indoor
        case .swiming:      return ImageRepo.SportTypes.swiming
        case .yoga:         return ImageRepo.SportTypes.yoga

        case .walk:         return ImageRepo.SportTypes.walk
        case .badminton:    return ImageRepo.SportTypes.badminton
        case .basketball:   return ImageRepo.SportTypes.basketball
        case .football:     return ImageRepo.SportTypes.football
        case .pingPong:     return ImageRepo.SportTypes.pingPong
        case .tennis:       return ImageRepo.SportTypes.tennis
        case .bike:         return ImageRepo.SportTypes.bike
        case .indoorRun:    return ImageRepo.SportTypes.indoorRun
        }
    }
    
    var band_z: SRSportType {
        switch self {
        case .normal:       return .normal
        case .run:          return .run
        case .climbing:     return .climbing
        case .indoor:       return .indoor
        case .swiming:      return .swiming
        case .yoga:         return .yoga

        case .walk:         return .walk
        case .badminton:    return .badminton
        case .basketball:   return .basketball
        case .football:     return .football
        case .pingPong:     return .pingPong
        case .tennis:       return .tennis
        case .bike:         return .bike
        case .indoorRun:    return .indoorRun
        }
    }
    
    var band_s: YFSportType {
        switch self {
        case .normal:       return .null
        case .run:          return .run
        case .climbing:     return .climbing
        case .indoor:       return .cyclingIndoor
        case .swiming:      return .swim
        case .yoga:         return .yoga

        case .walk:         return .walk
        case .badminton:    return .badminton
        case .basketball:   return .basketball
        case .football:     return .football
        case .pingPong:     return .pingPong
        case .tennis:       return .cyclingIndoor
        case .bike:         return .cycling
        case .indoorRun:    return .runningMachine
        }
    }
    
    static func parse(_ type: SRSportType) -> BDSportType {
        switch type {
        case .normal:       return .normal
        case .run:          return .run
        case .climbing:     return .climbing
        case .indoor:       return .indoor
        case .swiming:      return .swiming
        case .yoga:         return .yoga

        case .walk:         return .walk
        case .badminton:    return .badminton
        case .basketball:   return .basketball
        case .football:     return .football
        case .pingPong:     return .pingPong
        case .tennis:       return .tennis
        case .bike:         return .bike
        case .indoorRun:    return .indoorRun
        }
    }

    static func parse(_ type: YFSportType) -> BDSportType {
        switch type {
        case .null:             return .normal
        case .breathe:          return .walk //呼吸
        case .cycling:          return .bike //骑车或室外骑车
        case .cyclingIndoor:    return .indoor //室内骑车
        case .runningMachine:   return .indoorRun //跑步机或室内跑步
        case .run:              return .run //跑步或室外跑步
        case .swim:             return .swiming //游泳
        case .walk:             return .walk //走路
        case .weight:           return .indoor //举重
        case .yoga:             return .yoga //瑜咖
        case .badminton:        return .badminton //羽毛球
        case .basketball:       return .basketball //篮球
        case .skip:             return .indoor //跳绳
        case .exercise:         return .indoor //自由锻炼
        case .football:         return .football //足球
        case .climbing:         return .climbing //爬山
        case .pingPong:         return .pingPong //乒乓球
        }
    }
}

// MARK: 睡眠状态类型
enum BDSleepType {
    case none, deep, light, wake
    static func parse(_ type: SRSleepType) -> BDSleepType {
        switch type {
        case .none: return .none
        case .deep: return .deep
        case .light: return .light
        case .wake: return .wake
        }
    }
    
    static func parse(_ type: YFSleepType) -> BDSleepType {
        switch type {
        case .none, .wakeup: return .none
        case .deep: return .deep
        case .light: return .light
        case .start: return .wake
        }
    }
}

import NEUtils
enum BDAppNotifyType {
    case didEnterBackground
    case willEnterForeground
    case didFinishLaunching
    case didBecomeActive
    case willResignActive
    case didReceiveMemoryWarning
    case willTerminate
    case significantTimeChange
    case orientationDidChange
    
    static func parse(_ type: NEAppNotifyType) -> BDAppNotifyType {
        switch type {
        case .didEnterBackground: return .didEnterBackground
        case .willEnterForeground: return .willEnterForeground
        case .didFinishLaunching: return .didFinishLaunching
        case .didBecomeActive: return .didBecomeActive
        case .willResignActive: return .willResignActive
        case .didReceiveMemoryWarning: return .didReceiveMemoryWarning
        case .willTerminate: return .willTerminate
        case .significantTimeChange: return .significantTimeChange
        case .orientationDidChange: return .orientationDidChange
        }
    }
}

struct BDReaded {
    static var firmware:  Bool { SRFirmwareInfo.readed }
    static var userInfo:  Bool { SRUserInfo.readed }
    static var drinkInfo: Bool { SRDrinkInfo.readed }
    static var clockInfo: Bool {
        switch BandConnect.protocolType {
        case .series_z: return SRAlarmClock.readed
        case .series_s: return YFClockInfo.readed
        }
    }
    static var hybridInfo: Bool { YFHybridInfo.readed }
}


struct BDFirmwareFeature {
    
    /// 恢复出厂设置和查找手环
    var rest_find: Bool
    ///  天气推送
    var weather: Bool
    ///  拍照
    var camera: Bool
    ///  喝水提醒
    var waterRemind: Bool
    ///  温度单位切换
    var tempUnit: Bool
    ///  日期显示格式、血压、亮屏设置
    var dateformat_detect_brightScrren: Bool
    
    static func parse(_ m: SRFirmwareFeature) -> BDFirmwareFeature {
        BDFirmwareFeature(rest_find: m.rest_find, weather: m.weather, camera: m.weather, waterRemind: m.waterRemind, tempUnit: m.tempUnit, dateformat_detect_brightScrren: m.dateformat_detect_brightScrren)
    }
}


///: -  MARK: -  agps 运动

/// 训练运动状态
enum BDSportStatus: UInt8 {
    case stop   = 0 //停止
    case start      //开始
    case pause      //暂停
    case play       //继续
    case stopForce  //强制停止
    case syncData   //运动数据同步请求
    
    var band_s: YFSportStatus {
        switch self {
        case .stop: return .stop
        case .start: return .start
        case .pause: return .pause
        case .play: return .play
        case .stopForce: return .stopForce
        case .syncData: return .syncData
        }
    }
}

struct BDSportAgps {
    /// 运动类型
    public var type: BDSportType = .walk
    /// 运动状态
    public var status: BDSportStatus = .stop
    /// 时间 (s)
    public var time: Int32 = 0
    /// 距离 (m)
    public var distance: Int32 = 0
    
    var band_s: YFSportAgps {
        let m = YFSportAgps()
        m.type = type.band_s
        m.status = status.band_s
        m.time = time
        m.distance = distance
        return m
    }
}
