//
//  BandCalculator.swift
//  CTFit
//
//  Created by Mac on 2021/10/18.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation

/// [0.0 ~ 1.0]
func random() -> Float {
    return Float(drand48());
}

/// [min ~ max]
func random(min: Int, max: Int) -> Int {
    return Int(truncatingIfNeeded: arc4random_uniform(UInt32(max - min + 1)) + UInt32(min))
}

public typealias BDCalculatorType = Double

public struct BandCalculator {

    public struct Distance {
        
        public struct Foot {
            public static let inch: BDCalculatorType = 12
        }
        /// 1 inch = 2.54 cm
        public struct Inch {
            public static let cm:   BDCalculatorType = 2.54
        }
        public struct Cm {
            public static let inch: BDCalculatorType = 1/Inch.cm
        }
        
        /// 1 mile = 1.609344 km
        public struct Mile {
            public static let foot: BDCalculatorType   = 5280
            public static let km:   BDCalculatorType   = 1.609344
            public static let m:    BDCalculatorType   = 1609.344
        }

        /// 设备传回的距离单位为 1 m
        public struct Device {
            public static let m:    BDCalculatorType   = 1
            public static let km:   BDCalculatorType   = 0.001
            public static let mile: BDCalculatorType   = km / Mile.km
        }
        
        /// 设备传回的目标距离单位为 1 m
        public struct Goal {
            public static let m:    BDCalculatorType   = 1
            public static let km:   BDCalculatorType   = 0.001
            public static let mile: BDCalculatorType   = km / Mile.km
        }
    }
    
    public struct Weight {
        /// 1 kg = 2.20462262 lb
        public struct Kg {
            public static let lb:  BDCalculatorType    = 2.20462262
        }
        public struct Lb {
            public static let kg:  BDCalculatorType    = 1/Kg.lb
        }
    }

}
/// 1、Math.floor()向下取整，即小于这个数的最大的那个整数;
/// 2、Math.ceil()向上取整，即大于这个数的最小的那个整数；
/// 3、Math.rint()返回最接近该值的那个整数 0.5: -> 0 、 -0.5: -> 0
/// 4、Math.round()“四舍五入” -- Math.floor(x + 0.5)
extension BandCalculator {
    
    // MARK: - 身高 、 步幅 单位换算
    public static func imperial(cm: Int) -> (foot: Int, inch: Int) {
        let inch = lround(BDCalculatorType(cm) / Distance.Inch.cm)
        return (inch / Int(Distance.Foot.inch),  inch % Int(Distance.Foot.inch))
    }
    public static func cm(imperial: (foot: Int, inch: Int)) -> Int {
        let inch = (BDCalculatorType(imperial.foot) * Distance.Foot.inch + BDCalculatorType(imperial.inch))
        return  lround(inch * Distance.Inch.cm)
    }
    
    
    // MARK: - 体重换算
    public static func lb(kg: BDCalculatorType) -> BDCalculatorType {
        return kg * Weight.Kg.lb
    }
    public static func kg(lb: BDCalculatorType) -> BDCalculatorType {
        return lb * Weight.Lb.kg
    }
    
    // MARK: - 距离换算
    public static func mile(km: BDCalculatorType) -> BDCalculatorType {
        return km / Distance.Mile.km
    }
    public static func mile(m: BDCalculatorType) -> BDCalculatorType {
        return m / Distance.Mile.m
    }
    
    /// device: 设备记录距离值单位
    public static func km(device: BDCalculatorType) -> BDCalculatorType {
        return device * Distance.Device.km
    }
    public static func mile(device: BDCalculatorType) -> BDCalculatorType {
        return device * Distance.Device.mile
    }
    
    /// goal：目标距离单位
    public static func km(goal: BDCalculatorType) -> BDCalculatorType {
        return goal * Distance.Goal.km
    }
    public static func mile(goal: BDCalculatorType) -> BDCalculatorType {
        return goal * Distance.Goal.mile
    }
    public static func goal(km: BDCalculatorType) -> BDCalculatorType {
        return km / Distance.Goal.km
    }
    public static func goal(mile: BDCalculatorType) -> BDCalculatorType {
        return mile / Distance.Goal.mile
    }
    
    // MARK: - 温度
    public static func fahrenheit(celsius: BDCalculatorType) -> BDCalculatorType {
        return  32 + (1.8 * celsius)
    }
}
