//
//  BandCommand.swift
//  CTFit
//
//  Created by Mac on 2021/10/8.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation
import RxSwift
import RxCocoa
import Repeater
import RxBluetoothKit
import BandKit
import YFitKit

public typealias BDWriteClosure = ((Error?) -> Void)

extension SRRequestError {
    public var locastr: String? {
        switch self {
        case .bleoff: return SRString.Base.ble_off.locastr
        case .disconnected: return SRString.Base.disconnect_band.locastr
        case .sendfailed: return SRString.Base.disconnect_band.locastr
        case .timeout: return SRString.Base.request_timeout.locastr
        default : return nil
        }
    }
}

extension SRReponseError {
    var locastr: String? {
        switch self {
        case .delay: return nil
        default: return SRString.Base.saved_failed.locastr
        }
    }
}

enum YFReponseError: Int {
    case sendFailed = -2000
    case overTime
    case bleClosed
    case deviceDisconnect
    case bleExceptional
    
    var locastr: String? {
        switch self {
        case .bleClosed: return SRString.Base.ble_off.locastr
        case .deviceDisconnect: return SRString.Base.disconnect_band.locastr
        case .overTime: return SRString.Base.request_timeout.locastr
        default: return SRString.Base.saved_failed.locastr
        }
    }
}


public enum BDFeatureState: UInt8 {
    case off = 0x00
    case on  = 0x01
}

class BandCommand {
    
    private var protocolType: BDProtocolType { BandConnect.protocolType }
    
    private func send(_ cmd: SRCommand, _ toastError: Bool = false, callback: BDWriteClosure? = nil) {
        if toastError {
            ToastUtils.showActivity()
            cmd.writeToDevice({ (value, error) in
                ToastUtils.hideActivity()
                if let hasError = error as? SRRequestError, let msg = hasError.locastr { ToastUtils.showToastCCV(error: msg) }
                if let hasError = error as? SRReponseError, let msg = hasError.locastr { ToastUtils.showToastCCV(error: msg) }
                if let err = error, case SRRequestError.noreponse = err { callback?(nil) }
                else { callback?(error) }
            })
        } else {
            cmd.writeToDevice { callback?($1) }
        }
    }
    
    private func send(_ cmd: YFCommand, _ toastError: Bool = false, callback: BDWriteClosure? = nil) {
        if toastError {
            ToastUtils.showActivity()
            cmd.writeToDevice({ (error) in
                ToastUtils.hideActivity()
                if let err = error,
                   let hasError = YFReponseError(rawValue: err._code),
                   let msg = hasError.locastr { ToastUtils.showToastCCV(error: msg) }
                callback?(error)
            })
        } else {
            cmd.writeToDevice { callback?($0) }
        }
    }
}


extension BandCommand {
    
    ///: - Read decrds
    private func readRecordDetailToday(with type: BDRecordType) {
        if protocolType == .series_z {
            send(SRCommand.readRecordDetail(type.band_z, .t0))
        }
    }
    
    private func readDetectRecords(){
        if protocolType == .series_z {
            send(SRCommand.readDetectRecords())
        }
    }

    ///: - Find band
    private func findBand(_ callback: BDWriteClosure? = nil) {
        switch protocolType {
        case .series_z: send(SRCommand.findBandEvent(), true) { error in
            guard error == nil else { return }
            guard BandConnect.protocolType == .series_z else { return }
            AlertManager.show(title: nil, message: SRString.Connected.findband_alert.locastr, okHander: { (_) in })
        }
        case .series_s: send(YFCommand.findBand(.on), true, callback: callback);
        }
    }
    
    ///: - Reset
    private func rest() {
        switch protocolType {
        case .series_z: send(SRCommand.restFactory(), true)
        case .series_s: send(YFCommand.reset() , true);
        }
    }
    
    ///: - Camera
    private func camera(for state: BDFeatureState) {
        switch protocolType {
        case .series_z:
            switch state {
            case .on: send(SRCommand.cameraEvent(0))
            case .off: send(SRCommand.cameraEvent(1))
            }
        case .series_s:
            switch state {
            case .on: send(YFCommand.camera(.on));
            case .off: send(YFCommand.camera(.off));
            }
        }
    }
    
    ///: - Agps
    private func agpsStart(for type: BDSportType) {
        if protocolType == .series_z {
            send(SRCommand.exerciseStart(type.band_z))
        }
    }

    private func agpsEnd() {
        if protocolType == .series_z {
            send(SRCommand.exerciseEnd())
        }
    }
        
    private func agpsSync(_ sport: YFSportAgps) {
        if protocolType == .series_s {
            send(YFCommand.agps(sport))
        }
    }
    
    private func agpsEventStart() {
        if protocolType == .series_z {
            send(SRCommand.gpsEvent(0))
        }
    }

    private func agpsEventEnd() {
        if protocolType == .series_z {
            send(SRCommand.gpsEvent(1))
        }
    }
    
    ///: - Detect for custom
    private func detect(_ type: BDDetectType = .bpm, callback: BDWriteClosure? = nil){
        if protocolType == .series_z {
            send(SRCommand.detect(type.band_z), true, callback: callback)
        }
    }
    
    ///: - Clock for save
    private func save(_ clockInfo: BDClockInfo, callback: BDWriteClosure? = nil){
        guard BDClockInfo.band != clockInfo else { return }
        if protocolType == .series_z {
            let newInfo = clockInfo.band_z
            guard SRDataManager.clockInfo != newInfo else { return }
            send(SRCommand.setAlarmClock(newInfo), true) { error in
                guard error == nil else { return }
                SRDataManager.clockInfo = newInfo
                ToastUtils.showToast(success: SRString.Clock.saved_clock_success.locastr)
            }
        }
        
        if protocolType == .series_s {
            let newInfo = clockInfo.band_s
            send(YFCommand.setClockInfo(newInfo), true) { error in
                guard error == nil else { return }
                YFDataManager.hybridInfo.update(newInfo)
                ToastUtils.showToast(success: SRString.Clock.saved_clock_success.locastr)
            }
        }
    }
}

extension BandCommand {
    private static let shared: BandCommand = BandCommand()
    
    ///: - Read decrds
    static func readRecordDetailToday(with type: BDRecordType) { shared.readRecordDetailToday(with: type) }
    static func readDetectRecords() { shared.readDetectRecords() }
    
    ///: - Event commond
    static func findBand(callback: BDWriteClosure? = nil) { shared.findBand(callback) }
    static func rest() { shared.rest() }
    static func camera(for state: BDFeatureState) { shared.camera(for: state) }
    
    ///: - Agps
    static func agpsStart(for type: BDSportType) { shared.agpsStart(for: type) }
    static func agpsEnd() { shared.agpsEnd() }
    static func agpsEventStart() { shared.agpsEventStart() }
    static func agpsEventEnd() { shared.agpsEventEnd() }
    static func agpsSync(_ sport: BDSportAgps) { shared.agpsSync(sport.band_s) }
    
    ///: - Detect for custom
    static func detect(_ type: BDDetectType = .bpm, callback: BDWriteClosure? = nil) { shared.detect(type, callback: callback) }

    ///: - Clock for save
    static func save(_ clockInfo: BDClockInfo, callback: BDWriteClosure? = nil) { shared.save(clockInfo, callback: callback) }
    
}

