//
//  BDBandConnect.swift
//  CTFit
//
//  Created by Mac on 2021/9/29.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation
import RxSwift
import Repeater
import RxBluetoothKit
import BandKit
import YFitKit

enum BDProtocolType: Int, BDEnumtProtocol {
    case series_z = 0
    case series_s = 1
}

enum BDConnectAction {

    case success, failed(FailedType)

    enum FailedType {
        case bleoff, timeout, code(Int)
        var text: String {
            switch self {
            case .bleoff: return SRString.Base.ble_off.locastr
            case .timeout: return SRString.Searched.connect_timeout.locastr
            case .code(let code):
                if code == 14 { return SRString.Searched.pairlost_alert.locastr }
                return SRString.Searched.connect_failed.locastr
            }
        }
    }
    
    static func parse(_ state: SRConnectEstablishState) -> BDConnectAction {
        switch state {
        case .successed: return .success
        case .failed(let type):
            switch type {
            case .bleoff: return .failed(.bleoff)
            case .timeout: return .failed(.timeout)
            case .code(let code): return .failed(.code(code))
            }
        }
    }
}

enum BDConnectState {
    case connected, disconnected
}

enum BDBleState {
    case unknown
    case resetting
    case unsupported
    case unauthorized
    case poweredOff
    case poweredOn
    
    static func parse(_ t: BluetoothState) -> BDBleState {
        switch t {
        case .unknown:      return .unknown
        case .resetting:    return .resetting
        case .unsupported:  return .unsupported
        case .unauthorized: return .unauthorized
        case .poweredOff:    return .poweredOff
        case .poweredOn:    return .poweredOn
        }
    }
    
}


class BandConnect {
    
    init() { observe() }
    
    // MARK: - Field
    private(set) var band: BandPeripheral?
   
    //Rx
    private lazy var disposeBag = DisposeBag()
    private lazy var mainScheduler = MainScheduler.instance

    /// : - Private Subject
    private var actionConnecting: Bool = false
    private let statusSubject = PublishSubject<BDConnectState>()
    private let actionSubject = PublishSubject<BDConnectAction>()
    private let bleStateSubject = PublishSubject<BDBleState>()


    ///: - series_s 连接时 超时定时器
    private lazy var yftimer: Repeater = {
        let repeater = Repeater.once(after: .seconds(10.0), queue: DispatchQueue.main) { [weak self] (_) in
            guard let strong = self else { return }
            YFKitService.disconnect()
            strong.onConnectAction(protocol: .series_s, connect: .failed(.timeout))
        }
        repeater.pause()
        return repeater
    }()
    
    ///: - 当前协议类型，在连接设备时确认，默认为 series_z
    private let key = "kLatestProtoclTypeKey";
    private var _protocolType: BDProtocolType? {
        didSet {
            guard _protocolType != oldValue else { return }
            UserDefaults.standard.setValue(_protocolType?.rawValue, forKey: key)
            UserDefaults.standard.synchronize()
        }
    }
    private var protocolType: BDProtocolType {
        get {
            if let value = _protocolType { return value }
            if let _ = UserDefaults.standard.value(forKey: key) {
                let number = UserDefaults.standard.integer(forKey: key)
                let value = BDProtocolType(rawValue: number) ?? .series_z
                _protocolType = value
                return value
            } else {
                _protocolType = .series_z
                return .series_z
            }
        }
        set {
            _protocolType = newValue
        }
    }
    
    
    ///: - 蓝牙状态
    private var centralState: BluetoothState { SRKitService.centralState }
    
    ///： -  是否已绑定设备
    private var binded: Bool {
        guard SRKitService.binded || YFKitService.binded else { return false }
        return true
    }
    
    ///： -  是否已连接
    private var isConnected: Bool {
        if protocolType == .series_z { return SRKitService.isConnected }
        if protocolType == .series_s { return YFKitService.isConnected }
        return false
    }

    ///: -  是否自动连接
    private var autoConnecting: Bool {
        get {
            if protocolType == .series_z { return SRKitService.autoConnecting }
            if protocolType == .series_s { return YFKitService.autoConnecting }
            return false
        }
        set {
            if protocolType == .series_z { SRKitService.autoConnecting = newValue }
            if protocolType == .series_s { YFKitService.autoConnecting = newValue }
        }
    }
    
    ///: -  当前连接设备
    private var currentPeripheral: BandPeripheral? {
        if protocolType == .series_z, let p = SRKitService.currentPeripheral { return BandPeripheral(searched: p) }
        if protocolType == .series_s, let p = YFKitService.currentPeripheral { return BandPeripheral(searched: p) }
        return nil
    }
    
    ///: - 设备名
    private var peripheralName: String {
        guard let hasPeripheral = currentPeripheral else { return "Unknown Name" }
        return hasPeripheral.peripheralName
    }
}


/// : - 连接
extension BandConnect {
    private func connect(for band: BandPeripheral) {
        actionConnecting = true
        self.band = band
        if band.protocolType == .series_z { connect(for: band.peripheral_z!) }
        if band.protocolType == .series_s { connect(for: band.peripheral_s!) }
    }
    
    private func disconnect() {
        SRKitService.disconnect()
        YFKitService.disconnect()
    }
    
    private func connect(for perpheral: SRPeripheral) {
        SRKitService.connect(for: perpheral, time: 10)
    }
    
    private func connect(for perpheral: YFPeripheral) {
        yftimer.reStart()
        YFKitService.connect(for: perpheral)
    }
    
    private func reconnect() {
        SRKitService.reconnectBindPeripheral()
        YFKitService.reconnectBindPeripheral()
    }

    
    private func bind() {
        if protocolType == .series_z { SRKitService.bind() }
        if protocolType == .series_s { YFKitService.bind() }
    }
    
    private func unbind() {
        SRKitService.unbind()
        YFKitService.unbind()
    }
}


/// : - 连接状态更改
extension BandConnect {
    
    private func observe(){
        SRKitService.bluetoothStateOutput.observeOn(mainScheduler).subscribe(onNext: { [weak self] state in
            guard let strong = self else { return }
            strong.onBleState(BDBleState.parse(state))
        }).disposed(by: disposeBag)
        
        /// series_z
        SRKitService.connectionEstablishOutput.observeOn(mainScheduler).subscribe(onNext: { [weak self] state in
            guard let strong = self else { return }
            strong.onConnectAction(protocol: .series_z, connect: BDConnectAction.parse(state))
        }).disposed(by: disposeBag)
        
        SRKitService.connectionStateOutput.observeOn(mainScheduler).subscribe(onNext: { [weak self] state in
            guard let strong = self else { return }
            switch state {
            case .connected: strong.onConnectState(protocol: .series_z, connect: .connected)
            case .disconnected(_):  strong.onConnectState(protocol: .series_z, connect: .disconnected)
            }
        }).disposed(by: disposeBag)
        
        /// series_s
        YFKitService.productStatusOutput.observeOn(mainScheduler).subscribe(onNext: { [weak self] status in
            guard let strong = self else { return }
            switch status {
            case .searching, .connecting, .completed: break
            case .connected:
                strong.onConnectState(protocol: .series_s, connect: .connected)
                strong.onConnectAction(protocol: .series_s, connect: .success)

            case .powerOff:
                strong.onConnectState(protocol: .series_s, connect: .disconnected)
                strong.onConnectAction(protocol: .series_s, connect: .failed(.bleoff))
            
            default:
                strong.onConnectState(protocol: .series_s, connect: .disconnected)
                if BandConnect.centralState != .poweredOn {
                    strong.onConnectAction(protocol: .series_s, connect: .failed(.bleoff))
                } else {
                    strong.onConnectAction(protocol: .series_s, connect: .failed(.code(0)))
                }
            }
        }).disposed(by: disposeBag)
    }
    
    private func onConnectState(protocol type :BDProtocolType, connect state: BDConnectState ) {
        switch state {
        case .connected: protocolType = type
        default: break
        }
        statusSubject.onNext(state)
    }
    
    private func onConnectAction(protocol type :BDProtocolType, connect state: BDConnectAction ) {
        onActionOutput(state)
    }
    
    private func onActionOutput(_ state: BDConnectAction) {
        if actionConnecting {
            actionConnecting = false
            actionSubject.onNext(state)
            yftimer.pause()
        }
    }
    
    private func onBleState(_ state: BDBleState) {
        bleStateSubject.onNext(state)
    }
}


// MARK: -  Class for public
extension BandConnect {
    
    static let shared: BandConnect = BandConnect()
    public static var protocolType: BDProtocolType { shared.protocolType }

    ///: -  蓝牙状态
    public static var centralState: BluetoothState { shared.centralState }
    ///: -  是否已绑定设备
    public static var binded: Bool { shared.binded }
    ///: -  是否已连接
    public static var isConnected: Bool { shared.isConnected }
    ///: - 是否自动连接
    public static var autoConnecting: Bool { shared.autoConnecting }
    ///: - 当前连接设备
    public static var currentPeripheral: BandPeripheral? { shared.currentPeripheral }
    ///: - 设备名
    public static var peripheralName: String { shared.peripheralName }

    
    public static func connect(for band: BandPeripheral) { shared.connect(for: band) }
    public static func disconnect() { shared.disconnect() }
    public static func reconnect() { shared.reconnect() }
    
    public static func bind() { shared.bind() }
    public static func unbind() { shared.unbind() }

    /// : - Public Output
    public static var statusChangedOutput: Observable<BDConnectState> {
        return shared.statusSubject.asObservable()
    }
    
    public static var actionChangedOutput: Observable<BDConnectAction> {
        return shared.actionSubject.asObservable()
    }
    
    public static var bleStateOutput: Observable<BDBleState> {
        return shared.bleStateSubject.asObservable()
    }
}

