//
//  BandSearch.swift
//  CTFit
//
//  Created by Mac on 2021/9/29.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation
import RxSwift
import BandKit
import YFitKit

enum BDSearchAction {
    enum EndType {
        case bleoff, complet
    }
    case found(BandPeripheral), end(EndType)
}

class BandSearch {
    
    private init() {
        observe()
    }
 
    //Rx
    private lazy var disposeBag = DisposeBag()
    private lazy var mainScheduler = MainScheduler.instance

    /// : - Private Subject
    private let searchActionSubject = PublishSubject<BDSearchAction>()
}

extension BandSearch {
    private func observe() {
        SRKitService.scanningOutput.subscribe(onNext: { [unowned self] result in
            if case SRResult.success(let peripheral) = result {
                self.searchActionSubject.onNext(.found(BandPeripheral(searched: peripheral)))
            }
            if case SRResult.error(_) = result {
                self.searchActionSubject.onNext(.end(.complet))
                self.searchStop()
            }
        }).disposed(by: disposeBag)
        
        YFKitService.foundDeviceOutput.subscribe(onNext: { [unowned self] peripherals in
            peripherals.forEach { peripheral in
                self.searchActionSubject.onNext(.found(BandPeripheral(searched: peripheral)))
            }
        }).disposed(by: disposeBag)
    }
    
    private func searchStart(_ time: Int) {
        guard SRKitService.centralState == .poweredOn else {
            searchActionSubject.onNext(.end(.bleoff))
            return
        }
        SRKitService.startScanning(time)
        YFKitService.searchDeviceStart()
    }
    
    private func searchStop() {
        SRKitService.stopScanning()
        YFKitService.searchDeviceStop()
    }
    
    private func retrieveConnectedPeripherals() -> [BandPeripheral] {
        var peripherals: [BandPeripheral] = []
        let peripherals_z = SRKitService.retrieveConnectedPeripherals()
        if peripherals_z.count > 0 { peripherals += peripherals_z.map({ BandPeripheral(searched: $0) }) }
        let peripherals_s = SRKitService.retrieveConnectedPeripherals_s()
        if peripherals_s.count > 0 { peripherals += peripherals_s.map({ BandPeripheral(searched: YFPeripheral($0.peripheral))}) }
        return peripherals
    }

}

extension BandSearch {
    static let shared = BandSearch()
    
    /// : - Private Subject
    static var searchActionOutput: Observable<BDSearchAction> { shared.searchActionSubject.asObserver().observeOn(MainScheduler.asyncInstance) }

    static func searchStart(_ time: Int = 3) { shared.searchStart(time) }
    static func searchStop() { shared.searchStop() }
    static func retrieveConnectedPeripherals() -> [BandPeripheral] { shared.retrieveConnectedPeripherals() }

}
