//
//  BandSettings.swift
//  CTFit
//
//  Created by Mac on 2021/10/28.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation

class BandSettings: BDModelProtocol {
    required init() { }

    ///:  - cache feild
    var user: BDUser = BDUser()
    var format: BDFormat = BDFormat()
    
    /// 目标
    var goal: BDGoal = BDGoal()
    /// 目标铃声
    public var reachRing: BDReachRing = BDReachRing()
    /// 运动心率报警
    var heartRateAlarm: BDAlarmExercise = BDAlarmExercise()
    /// 消息推送应用
    var ancsList: [BDAncsAppType] = BDAncsAppType.all

    
    // MARK: - Feature
    /// 翻腕亮屏
    var rollBrightScrren = BDRemind(type: .raiseLight, enable: true, startHour: 0, endHour: 23)
    /// 静态心率 检测
    var restingHeartRate: BDRemind = BDRemind(type: .heart, enable: false, startHour: 0, endHour: 23)
    
    // MARK: - Remind
    /// 来电提醒
    var callRemind: BDRemind = BDRemind(type: .call, enable: true, startHour: 0, endHour: 23)
    /// 消息提醒
    var messageRemind: BDRemind = BDRemind(type: .message, enable: true, startHour: 0, endHour: 23)
    /// 久坐提醒
    var sedentaryRemind: BDRemind = BDRemind(type: .sedentary, startHour: 9, endHour: 18)
    /// 喝水提醒
    var waterRemind: BDRemind = BDRemind(type: .water, startHour: 8, endHour: 20)
    
    /// 勿扰模式
    var noDistrub: BDRemind = BDRemind(type: .nodistrub, startHour: 23, endHour: 8)
    var targetRemind: BDRemind = BDRemind(type: .goal)

}

extension BandSettings: Equatable {
    public static func == (lhs: BandSettings, rhs: BandSettings) -> Bool {
        (lhs.jsonString == rhs.jsonString)
    }
}

///: - 数据缓存
extension BandSettings {
    private static var tempObject = BandSettings()
    private static let key = "BandSettingsInfoKey"
    private static var _standard: BandSettings?
    static var standard: BandSettings {
        get {
            if let info = _standard { return info}
            guard let jsonString = UserDefaults.standard.string(forKey: key),
                  let info = BandSettings.deserialize(from: jsonString) else {
                let model = BandSettings()
                _standard = model
                return model
            }
            _standard = info
            return info
        }
        set {
            _standard = newValue
            UserDefaults.standard.set(newValue.jsonString, forKey: key)
            UserDefaults.standard.synchronize()
        }
    }
}



extension BandSettings {
    
    static func extract() -> BandSettings{
        let info = standard
        info.loadForBand()
        tempObject = info.copy()
        return info.copy()
    }
    
    ///: - Method
    func loadForBand() {
        switch BandSettings.protocolType {
        case .series_z: loadForSeriesZ()
        case .series_s: loadForSeriesS()
        }
    }
    
    func saveToBand() {
        guard BandSettings.tempObject != self else { return }
        switch BandSettings.protocolType {
        case .series_z: saveForSeriesZ()
        case .series_s: saveForSeriesS()
        }
    }
    
    func saveToCache() {
        BandSettings.standard = self.copy()
    }
}


///: - 类方便属性，外部判断使用
extension BandSettings {
    /// 协议类型
    fileprivate static var protocolType: BDProtocolType { BandConnect.protocolType }
    
    /// 是否为英制
    static var isImperial: Bool { standard.format.system == .imperial }
    
    static var goal: BDGoal { standard.goal }

    static var format: BDFormat { standard.format }
    
    /// 惯用手
    static var handEnable: Bool {
        if protocolType == .series_s { return true}
        return false
    }
    
    /// 免打扰
    static var nodisturb: Bool {
        if protocolType == .series_s { return true }
        return false
    }
    
    /// 血压检测、血压参考值
    static var bpEnable: Bool {
        if protocolType == .series_z { return bpEnable_z }
        return false
    }
    
    /// 喝水提醒
    static var waterRemindEnable: Bool {
        if protocolType == .series_z { return waterRemindEnable_z }
        if protocolType == .series_s  { return true }

        return false
    }
    
    /// 温度单位
    static var tempUnitEnable: Bool {
        if protocolType == .series_z { return tempUnitEnable_z }
        return false
    }
    
    ///  日期显示格式、血压、亮屏设置
    static var dateformat_detect_brightScrren: Bool {
        if protocolType == .series_z { return dateformat_detect_brightScrren_z }
        return false
    }
}

import BandKit
extension BandSettings {
    
    private static var firmwareInfo: SRFirmwareInfo { SRDataManager.firmwareInfo }
    
    private static var isAbys: Bool { SRDataManager.userInfo.isAbys }

    /// 血压检测、血压参考值
    private static var bpEnable_z: Bool { firmwareInfo.feature.dateformat_detect_brightScrren }
    
    /// 喝水提醒
    private static var waterRemindEnable_z: Bool { (firmwareInfo.feature.waterRemind || isAbys) }
    
    /// 温度单位
    private static var tempUnitEnable_z: Bool { firmwareInfo.feature.tempUnit }
    
    ///  日期显示格式、血压、亮屏设置
    private static var dateformat_detect_brightScrren_z: Bool { (firmwareInfo.feature.dateformat_detect_brightScrren || isAbys) }
    
}



