//
//  BandSettingsModels.swift
//  CTFit
//
//  Created by Mac on 2021/10/28.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation
import BandKit
import YFitKit

///: - 铃声
class BDVibrate: BDModelProtocol {
    required init() { }
    init(type: BDVibrateType = .staccato, count: UInt8 = 25) {
        self.type = type
        self.count = count
    }
    /// 震动类型： 8 种
    public var type: BDVibrateType = .staccato
    /// 重复次数：( 0 ~ 31 )
    public var count: UInt8 = 25
    
    func update(_ vibrate: SRVibrate) {
        type = BDVibrateType.parse(vibrate.type)
        count = vibrate.count
    }
    
    var band_z: SRVibrate { SRVibrate(type: type.band_z, count: count) }
}

extension BDVibrate: Equatable {
    static func == (lhs: BDVibrate, rhs: BDVibrate) -> Bool {
        guard lhs.type == rhs.type, lhs.count == rhs.count else { return false }
        return true
    }
}


///: - 格式参数
class BDFormat: BDModelProtocol {
    required init() { }
    /// 小时制 , 默认（h24）
    var hourly: BDHourlyType = .h24
    ///  公英制 — 默认（metric）
    var system: BDSystemType = .metric
    ///  温度单位 — 默认（centigrade）
    var tempUnit: BDTempUnitsType = .centigrade
    /// 日期显示格式:  0（ 默认值 — MMDD ）， 1 ... 其它
    var dateFormat: BDDateformatType = .mmdd
    /// 亮屏时间
    var brightScreenTime: BDBrightScreenTimeType = .short
    

    ///: - series_z
    func update(_ userInfo: SRUserInfo) {
        hourly = BDHourlyType.parse(userInfo.format.hourly)
        system = BDSystemType.parse(userInfo.format.system)
        tempUnit = BDTempUnitsType.parse(userInfo.format.tempUnit)
        brightScreenTime = BDBrightScreenTimeType.parse(userInfo.brightScreenTime)
    }
    
    func update(_ drinkInfo: SRDrinkInfo) {
        dateFormat = BDDateformatType.parse(drinkInfo.dateFormat)        
    }
    
    var band_z: SRFormat {
        let m = SRDataManager.userInfo.format.copy()
        m.hourly = hourly.band_z
        m.system = system.band_z
        m.tempUnit = tempUnit.band_z
        return m
    }
    
    ///: - series_s
    func update(_ hybridInfo: YFHybridInfo) {
        hourly = BDHourlyType.parse(hybridInfo.hourlyInfo.format)
        system = BDSystemType.parse(hybridInfo.unitInfo.system)
    }
    
    var hourlyInfo: YFHourly {
        let m = YFHourly()
        m.format = hourly.band_s
        return m
    }
    
    var unitInfo: YFUnit {
        let m = YFUnit()
        m.system = system.band_s
        return m
    }

}

///: - 达标参数
class BDGoal: BDModelProtocol, Equatable {
    required init() {}
    init(step: UInt32 = 8000, calories: UInt32 = 500, distance: UInt32 = 4000) {
        self.step = step
        self.calories = calories
        self.distance = distance
    }
    /// 步数
    var step: UInt32 = 8000
    /// 卡路里 ( 大卡==千卡 )
    var calories: UInt32 = 500
    /// 距离（m）
    var distance: UInt32 = 4000
    
    func update(_ userInfo: SRUserInfo) {
        step = UInt32(userInfo.goal.step)
        calories = UInt32(userInfo.goal.calories)
        distance = UInt32(userInfo.goal.distance) * 100
    }
    
    var band_z: SRGoal {
        SRGoal(step: UInt16(truncating: NSNumber(value: step)),
               calories: UInt16(truncating: NSNumber(value: calories)),
               distance: UInt8(truncating: NSNumber(value: distance/100)))
    }
    
    ///: -  卡路里目标区间外，默认值500
    func calories(vaild value: UInt32) -> UInt32 {
        guard value >= 200, value <= 2000 else { return 500}
        return value
    }
    
    func update(_ goalInfo: YFGoalInfo) {
        step = goalInfo.step
        calories =  calories(vaild: goalInfo.calorie)
        distance = goalInfo.distance
    }
    
    var band_s: YFGoalInfo {
        let m = YFGoalInfo()
        m.step = step
        m.calorie = calories
        m.distance = distance
        return m
    }

    public static func == (lhs: BDGoal, rhs: BDGoal) -> Bool {
        (lhs.step == rhs.step &&  lhs.calories == rhs.calories  && lhs.distance == rhs.distance)
    }
}

///: - 达标铃声
class BDReachRing: BDModelProtocol {
    required init() { }
    init(step: BDVibrate = BDVibrate(type: .staccato, count: 25),
         calories: BDVibrate = BDVibrate(type: .none, count: 0),
         distance: BDVibrate = BDVibrate(type: .none, count: 0)) {
        self.step = step
        self.calories = calories
        self.distance = distance
    }
    
    /// 步数
    var step: BDVibrate = BDVibrate(type: .staccato, count: 25)
    /// 卡路里
    var calories: BDVibrate = BDVibrate(type: .none, count: 0)
    /// 距离
    var distance: BDVibrate = BDVibrate(type: .none, count: 0)
    
    func update(_ reachRing: SRReachRing) {
        self.step.update(reachRing.step)
        self.calories.update(reachRing.calories)
        self.distance.update(reachRing.distance)
    }
    
    var band_z: SRReachRing {
        SRReachRing(step: step.band_z, calories: calories.band_z, distance: distance.band_z)
    }
}

///: - 运动模式心率报警
class BDAlarmExercise: BDModelProtocol {
    required init() { }
    /// 心率过高报警开
    var highestEnable: Bool = false
    /// 心率过低和过高都打开
    var lowestEnable: Bool = false
    /// 过高心率
    var heartRateHigh: UInt8 = 160
    /// 过低心率
    var heartRateLow: UInt8 = 90
    /// 震动
    var vibrate = BDVibrate()

    func detailText(isHigh: Bool = true) -> String {
        if isHigh {
            if !highestEnable { return SRString.Base.closed.locastr}
            return "\(heartRateHigh)" + SRString.Heart.bpm.locastr
        } else  {
            if !lowestEnable { return SRString.Base.closed.locastr}
            return "\(heartRateLow)" + SRString.Heart.bpm.locastr
        }
    }
    
    func update(_ userInfo: SRUserInfo) {
        highestEnable = userInfo.heartRateAlarm.highestEnable
        lowestEnable = userInfo.heartRateAlarm.lowestEnable
        heartRateHigh = userInfo.heartRateAlarm.heartRateHigh
        heartRateLow = userInfo.heartRateAlarm.heartRateLow
    }
    
    var band_z: SRAlarmExercise {
        let m = SRDataManager.userInfo.heartRateAlarm.copy()
        m.highestEnable = highestEnable
        m.lowestEnable = lowestEnable
        m.heartRateHigh = heartRateHigh
        m.heartRateLow = heartRateLow
        m.vibrate = vibrate.band_z
        return m
    }

}

///: - 用户
class BDUser: BDModelProtocol {
    required init() { }
    /// 头像
    var portraitPath: String? = nil
    /// 昵称
    var nickname: String? = nil
    
    /// 年龄
    var age: UInt8 = 25
    /// 性别
    var gender: BDGenderType = .male
    /// 身高（cm）
    var height: UInt8 = 180
    /// 体重（kg）
    var weight: UInt8 = 65
    /// 步幅（cm）—  走
    var strideWalk: UInt8 = 65
    /// 步幅（cm）—  跑
    var strideRun: UInt8 = 80
    /// 血压检查参考值:  高压
    var sbpReference: UInt8 = 110
    /// 血压检查参考值:  低压
    var dbpReference: UInt8 = 90
    /// 惯用手
    var handType: BDHandType = .right
    
    func update(_ userInfo: SRUserInfo) {
        age = userInfo.user.age
        weight = userInfo.user.weight
        strideWalk = userInfo.user.strideWalk
        strideRun = userInfo.user.strideRun
    }
    
    func update(_ drinkInfo: SRDrinkInfo) {
        gender = BDGenderType.parse(drinkInfo.gender)
        height = drinkInfo.height
        if SRDataManager.firmwareInfo.feature.dateformat_detect_brightScrren {
            sbpReference = drinkInfo.sbpReference
            dbpReference = drinkInfo.dbpReference
        }
    }
        
    var band_z: SRUser {
        let m = SRDataManager.userInfo.user.copy()
        m.age = age
        m.weight = weight
        m.strideWalk = strideWalk
        m.strideRun = strideRun
        return m
    }
    
    func update(_ hybridInfo: YFHybridInfo) {
        let userInfo = hybridInfo.userInfo
        age = userInfo.age
        gender = BDGenderType.parse(userInfo.sex)
        height = userInfo.userHeight
        weight = userInfo.userWeight
        handType = BDHandType.parse(userInfo.lrHand)
    }
    
    var band_s: YFUserInfo {
        let m = YFDataManager.hybridInfo.userInfo.copy()
        m.age = age
        m.userHeight = height
        m.userWeight = weight
        m.sex = gender.band_s
        m.lrHand = handType.band_s
        return m
    }
}

///: - 提醒
class BDRemind: BDModelProtocol {
    required init() { }
    convenience init(type: BDRemindType = .call, enable: Bool = false, enable_noon: Bool = false,
         startHour: UInt8 = 6, endHour: UInt8 = 23,
         vibrate: BDVibrate = BDVibrate(),
         interval: UInt8 = 60) {
        self.init()
        self.type = type
        self.enable = enable
        self.enable_noon = enable_noon
        self.startHour = startHour
        self.endHour = endHour
    }

    /// 提醒类型
    private(set) var type: BDRemindType = .call
    /// 开关
    var enable: Bool = false
    /// 午休开关
    var enable_noon: Bool = false

    /// 开始 ( 0 ~ 23 )
    var startHour: UInt8 = 8
    var startMinute: UInt8 = 0
    /// 结束( 0 ~ 23 )
    var endHour: UInt8 = 20
    var endMinute: UInt8 = 59

    /// 震动
    var vibrate: BDVibrate = BDVibrate()
    /// 间隔时间 ( 分钟 )，默认 60
    var interval: UInt8 = 60
    
    private var _feature: UInt8 {
        if enable_noon { return 2 }
        if enable { return 1}
        return 0
    }
    
    func detailText(hourly: BDHourlyType) -> String {
        if !enable { return SRString.Base.closed.locastr }
        if !type.hasTime { return SRString.Base.opened.locastr }
        return CalendarUtils.stringFormat(Int(startHour), 0, hourly) + "~" + CalendarUtils.stringFormat(Int(endHour), 59, hourly)
    }
    
    ///: - series_z
    func update(_ remind: SRFeature) {
        enable = remind.feature > 0
        enable_noon = remind.feature > 1
        startHour = remind.startHour
        endHour = remind.endHour
    }
    
    func update(_ remind: SRRemind) {
        enable = remind.feature > 0
        enable_noon = remind.feature > 1
        startHour = remind.startHour
        endHour = remind.endHour
        vibrate.update(remind.vibrate)
    }
    
    func update(_ remind: SRRemindInterval) {
        enable = remind.feature > 0
        enable_noon = remind.feature > 1
        startHour = remind.startHour
        endHour = remind.endHour
        vibrate.update(remind.vibrate)
        interval = remind.interval
    }
    
    var feature: SRFeature {
        SRFeature(feature: _feature, startHour: startHour, endHour: endHour )
    }
    
    var remind: SRRemind {
        SRRemind(feature: _feature, startHour: startHour, endHour: endHour, vibrate: vibrate.band_z )
    }
    
    var remindInterval: SRRemindInterval {
        SRRemindInterval(feature: _feature, startHour: startHour, endHour: endHour, vibrate: vibrate.band_z, interval: interval )
    }

    ///: - series_s
    
    func update(_ remind: YFRaiseLight) {
        enable = remind.onoff.boolValue
        startHour = remind.startHour
        endHour = remind.endHour
    }
    
    func update(_ remind: YFHeartRateMonitor) {
        enable = remind.onoff.boolValue
    }
    
    func update(_ remind: YFRemindCall) {
        enable = remind.onoff.boolValue
    }
    
    func update(_ remind: YFMessageNotify) {
        enable = remind.onoff.boolValue
    }
    
    func update(_ remind: YFRemindTarget) {
        enable = remind.onoff.boolValue
    }
    
    func update(_ remind: YFRemindSedentary) {
        enable = remind.isOpen.boolValue
        enable_noon = remind.noon_onoff.boolValue
        startHour = remind.startHour
        endHour = remind.endHour
    }
    
    func update(_ remind: YFRemindDrink) {
        enable = remind.isOpen.boolValue
        interval = remind.interval
        startHour = remind.startHour
        endHour = remind.endHour
    }
    
    func update(_ remind: YFDisturb) {
        enable = remind.switch_flag.boolValue
        guard let first = remind.disturbs.first else { enable = false;  return }
        startHour = first.startHour
        endHour = first.endHour
    }
    
    var remindRaiseLight: YFRaiseLight {
        let m = YFRaiseLight()
        m.onoff = enable ? .on : .off
        m.startHour = startHour
        m.startMin = startMinute
        m.endHour = endHour
        m.endMin = endMinute
        return m
    }
    
    var heartRateMonitor: YFHeartRateMonitor {
        let m = YFHeartRateMonitor()
        m.onoff = enable ? .on : .off
        return m
    }
    
    var remindCall: YFRemindCall {
        let m = YFRemindCall()
        m.onoff = enable ? .on : .off
        return m
    }
    
    var messageNotify: YFMessageNotify {
        let m = YFMessageNotify()
        m.onoff = enable ? .on : .off
        return m
    }
    
    var remindTarget: YFRemindTarget {
        let m = YFRemindTarget()
        m.onoff = enable ? .on : .off
        return m
    }
    
    var remindSedentary: YFRemindSedentary {
        let m = YFRemindSedentary()
        m.isOpen = enable ? .on : .off
        m.noon_onoff = enable_noon ? .on : .off
        m.startHour = startHour
        m.startMin = startMinute
        m.endHour = endHour
        m.endMin = endMinute
        return m
    }
    
    var remindDrink: YFRemindDrink {
        let m = YFRemindDrink()
        m.isOpen = enable ? .on : .off
        m.interval = interval
        m.startHour = startHour
        m.startMin = startMinute
        m.endHour = endHour
        m.endMin = endMinute
        return m
    }
    
    var disturb: YFDisturb {
        let first = YFDisturbTimeItem()
        first.startHour = startHour
        first.startMin = startMinute
        first.endHour = endHour
        first.endMin = endMinute
        
        let m = YFDisturb()
        m.switch_flag = enable ? .on : .off
        m.count = 1
        m.disturbs = [first]
        return m
    }
}


