//
//  BandSettingsProtocol.swift
//  CTFit
//
//  Created by Mac on 2021/10/28.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation
import RxSwift
import BandKit
import YFitKit

///: - 数据监听 & 分发
extension BandSettings {
    enum ObserveTyp {
        case system, tempUnit, goal
    }
    
    static var observable: Observable<[ObserveTyp]> {
        return subject.asObservable().observeOn(MainScheduler.instance)
    }
    static func bandObserve() {
        observeForSeriesS()
        observeForSeriesZ()
    }
    
    private static let disposeBag = DisposeBag()
    private static let subject = PublishSubject<[ObserveTyp]>()
    private static func onUpdate(_ options: [ObserveTyp]) {
        standard.saveToCache()
        subject.onNext(options)
    }
}

extension BandSettings {
    func loadForSeriesZ() {
        if SRUserInfo.readed { update(SRDataManager.userInfo) }
        if SRDrinkInfo.readed { update(SRDataManager.drinkInfo) }
    }
    func saveForSeriesZ() { saveForBandZ() }
    static func observeForSeriesZ() { observeForBandZ() }
    
    
    func loadForSeriesS() { if YFHybridInfo.readed { update(YFDataManager.hybridInfo) }  }
    func saveForSeriesS() { saveForBandS() }
    static func observeForSeriesS() { observeForBandS() }
}

// MARK: - series_z
extension BandSettings {
    private static func observeForBandZ() {
        BandData.updatedOutput.observeOn(MainScheduler.instance).subscribe(onNext: { (dataType) in
            switch dataType {
            case .userInfo: standard.update(SRDataManager.userInfo)
            case .drinkInfo: standard.update(SRDataManager.drinkInfo)
            default: break
            }
        }).disposed(by: disposeBag)
    }
    
    private func saveForBandZ() {
        var cmdCount = 0
        var resultCount = 0;
        var successCount = 0;
        var hasError = false
        let closure: SRWriteClosure = { (reponse, error) in
            guard !hasError else { return }
            resultCount += 1
            if error == nil {
                successCount += 1;
                if successCount == cmdCount{
                    ToastUtils.showToast(success: SRString.Base.saved_success.locastr)
                }
            } else {
                hasError = true
                if let err = error as? SRRequestError, let msg = err.locastr { ToastUtils.showToastCCV(error: msg) }
                if let err = error as? SRReponseError, let msg = err.locastr { ToastUtils.showToastCCV(error: msg) }
            }
        }
        
        if setUserInfo(closure) { cmdCount += 1 }
        if setDrinkInfo(closure) { cmdCount += 1 }
        if cmdCount > 0 { ToastUtils.showToast(normal: SRString.Base.saving.locastr, duration: 10.0) }
    }
    
    private func update(_ userInfo: SRUserInfo) {
        let temp = self.copy()
        user.update(userInfo)
        format.update(userInfo)

        goal.update(userInfo)
        reachRing.update(userInfo.reachRing)
        heartRateAlarm.update(userInfo)
        ancsList = BDAncsAppType.parse(userInfo.ancsBlacks.list)

        rollBrightScrren.update(userInfo.rollBrightScrren)
        restingHeartRate.update(userInfo.restingHeartRate)
        callRemind.update(userInfo.callRemind)
        messageRemind.update(userInfo.messageRemind)
        sedentaryRemind.update(userInfo.sedentaryRemind)
        if userInfo.isAbys {
            waterRemind.update(userInfo.waterRemind)
        }
        
        guard temp != self else { return }
        /// 有数据更改
        var options = [ObserveTyp]()
        if temp.format.system != format.system { options.append(.system) }
        if temp.format.tempUnit != format.tempUnit { options.append(.system) }
        if temp.goal != goal { options.append(.goal) }
        BandSettings.onUpdate(options)
    }
    
    private func update(_ drinkInfo: SRDrinkInfo) {
        let temp = self.copy()
        user.update(drinkInfo)
        format.update(drinkInfo)
        waterRemind.update(drinkInfo.waterRemind)
        sedentaryRemind.interval = drinkInfo.sedentaryInterval
        guard temp != self else { return }
        /// 有数据更改
        BandSettings.onUpdate([])

    }
    
    private func userInfo() -> SRUserInfo {
        let userInfo = SRDataManager.userInfo.copy()
        userInfo.user = user.band_z
        userInfo.format = format.band_z
        userInfo.brightScreenTime =  format.brightScreenTime.band_z

        userInfo.goal = goal.band_z
        userInfo.reachRing = reachRing.band_z
        userInfo.heartRateAlarm = heartRateAlarm.band_z
        userInfo.ancsBlacks.list = BDAncsAppType.parse(ancsList)

        userInfo.rollBrightScrren = rollBrightScrren.feature
        userInfo.restingHeartRate = restingHeartRate.feature
        userInfo.callRemind = callRemind.remind
        userInfo.messageRemind = messageRemind.remind
        userInfo.sedentaryRemind = sedentaryRemind.remind
        if userInfo.isAbys {
            userInfo.waterRemind = waterRemind.remindInterval
        }
        return userInfo
    }
    
    private func drinkInfo() -> SRDrinkInfo {
        let drinkInfo = SRDataManager.drinkInfo.copy()
        drinkInfo.gender = user.gender.band_z
        drinkInfo.height = user.height
        drinkInfo.dateFormat = format.dateFormat.band_z
        drinkInfo.sbpReference = user.sbpReference
        drinkInfo.dbpReference = user.dbpReference
        drinkInfo.waterRemind = waterRemind.remindInterval
        drinkInfo.sedentaryInterval = sedentaryRemind.interval
        return drinkInfo
    }
    
    private func setUserInfo(_ closure: @escaping SRWriteClosure) -> Bool {
        let newUserInfo = userInfo()
        guard newUserInfo !=  SRDataManager.userInfo else { return false }
        DispatchQueue.main.asyncAfter(deadline: .now()+0.1) {
            SRCommand.setUserInfo(newUserInfo).writeToDevice({ (reponse, error) in
                if error == nil {
                    SRDataManager.userInfo = newUserInfo
                    BandSettings.standard.update(newUserInfo)
                }
                closure(reponse, error)
            })
        }
        return true
    }
    
    private func setDrinkInfo(_ closure: @escaping SRWriteClosure) -> Bool{
        let newDrinkInfo = drinkInfo()
        guard newDrinkInfo !=  SRDataManager.drinkInfo else { return false }
        DispatchQueue.main.asyncAfter(deadline: .now()+0.1) {
            SRCommand.setDrinkInfo(newDrinkInfo).writeToDevice({ (reponse, error) in
                if error == nil {
                    SRDataManager.drinkInfo = newDrinkInfo
                    BandSettings.standard.update(newDrinkInfo)
                }
                closure(reponse, error)
            })
        }
        return true
    }
}


// MARK: - series_s
extension BandSettings {
    private static func observeForBandS() {
        BandData.updatedOutput.observeOn(MainScheduler.instance).subscribe(onNext: { (dataType) in
            switch dataType {
            case .hybridInfo:
                standard.update(YFDataManager.hybridInfo);
                standard.saveForBandS(toast: false)
            default: break
            }
        }).disposed(by: disposeBag)
    }
    
    private func update(_ hybridInfo: YFHybridInfo) {
        let temp = self.copy()

        format.update(hybridInfo)
        
        user.update(hybridInfo)
        goal.update(hybridInfo.goalInfo)
        
        noDistrub.update(hybridInfo.disturbInfo)
        restingHeartRate.update(hybridInfo.heartRateMonitor)
        rollBrightScrren.update(hybridInfo.remindRaiseLight)
        messageRemind.update(hybridInfo.messageNotifyInfo)
        ancsList = BDAncsAppType.parse(hybridInfo.remindNotifySwitch)

        callRemind.update(hybridInfo.remindCall)
        sedentaryRemind.update(hybridInfo.remindSedentary)
        waterRemind.update(hybridInfo.remindDrink)
        targetRemind.update(hybridInfo.remindTarget)
        
        guard temp != self else { return }
        /// 有数据更改
        var options = [ObserveTyp]()
        if temp.format.system != format.system { options.append(.system) }
        if temp.format.tempUnit != format.tempUnit { options.append(.system) }
        if temp.goal != goal { options.append(.goal) }
        BandSettings.onUpdate(options)
    }
    
    private func hybridInfo() -> YFHybridInfo {
        let hybridInfo = YFDataManager.hybridInfo.copy()
        hybridInfo.unitInfo = format.unitInfo
        hybridInfo.hourlyInfo = format.hourlyInfo
        
        hybridInfo.userInfo = user.band_s
        hybridInfo.goalInfo = goal.band_s
        
        hybridInfo.disturbInfo = noDistrub.disturb
        hybridInfo.heartRateMonitor = restingHeartRate.heartRateMonitor
        hybridInfo.remindRaiseLight = rollBrightScrren.remindRaiseLight
        hybridInfo.messageNotifyInfo = messageRemind.messageNotify
        hybridInfo.remindNotifySwitch = BDAncsAppType.parse(ancsList)

        hybridInfo.remindCall = callRemind.remindCall
        hybridInfo.remindSedentary = sedentaryRemind.remindSedentary
        hybridInfo.remindDrink = waterRemind.remindDrink
        hybridInfo.remindTarget = targetRemind.remindTarget
        return hybridInfo
    }
    
    private func saveForBandS(toast: Bool = true) {
        let newInfo = hybridInfo()
        guard newInfo != YFDataManager.hybridInfo else { return }
        
        var cmdCount = 0
        var resultCount = 0;
        var successCount = 0;
        var hasError = false
        let closure: YFWriteClosure = { (error) in
            resultCount += 1
            if resultCount == cmdCount { BandSettings.standard.loadForSeriesS() }
            if let err = error {
                if toast,
                   hasError == false,
                   let hasReponseError = YFReponseError(rawValue: err._code),
                   let msg = hasReponseError.locastr { ToastUtils.showToastCCV(error: msg) }
                hasError = true
            } else {
                successCount += 1;
                if toast, successCount == cmdCount { ToastUtils.showToast(success: SRString.Base.saved_success.locastr) }
            }
        }
        
        if setUnit(newInfo.unitInfo, closure) { cmdCount += 1 }
        if setHourly(newInfo.hourlyInfo, closure) { cmdCount += 1 }

        if setUserInfo(newInfo.userInfo, closure) { cmdCount += 1 }
        if setGoalInfo(newInfo.goalInfo, closure) { cmdCount += 1 }

        if setDisturb(newInfo.disturbInfo, closure) { cmdCount += 1 }
        if setHeartRateMonitor(newInfo.heartRateMonitor, closure) { cmdCount += 1 }
        if setRaiseLight(newInfo.remindRaiseLight, closure) { cmdCount += 1 }
        if setMessageNotify(newInfo.messageNotifyInfo, closure) { cmdCount += 1 }
        if setMessageNotifyType(newInfo.remindNotifySwitch, closure) { cmdCount += 1 }

        if setRemindCall(newInfo.remindCall, closure) { cmdCount += 1 }
        if setRemindTarget(newInfo.remindTarget, closure) { cmdCount += 1 }
        if setRemindSedentar(newInfo.remindSedentary, closure) { cmdCount += 1 }
        if setRemindDrink(newInfo.remindDrink, closure) { cmdCount += 1 }
        if cmdCount > 0 { ToastUtils.showToast(normal: SRString.Base.saving.locastr, duration: 10.0) }

    }
    
    ///: -
    /// 设置单位
    private func setUnit(_ newInfo: YFUnit, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.unitInfo else { return false }
        YFCommand.setUnit(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置时间制
    private func setHourly(_ newInfo: YFHourly, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.hourlyInfo else { return false }
        YFCommand.setHourly(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    
    ///: -
    /// 用户信息
    private func setUserInfo(_ newInfo: YFUserInfo, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.userInfo else { return false }
        YFCommand.setUserInfo(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置目标提醒值
    private func setGoalInfo(_ newInfo: YFGoalInfo, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.goalInfo else { return false }
        YFCommand.setGoalInfo(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    
    ///: -
    /// 设置免打扰时间段
    private func setDisturb(_ newInfo: YFDisturb, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.disturbInfo else { return false }
        YFCommand.setDisturb(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置静态心率检测，只有开关
    private func setHeartRateMonitor(_ newInfo: YFHeartRateMonitor, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.heartRateMonitor else { return false }
        YFCommand.setHeartRateMonitor(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置翻腕亮屏
    private func setRaiseLight(_ newInfo: YFRaiseLight, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.remindRaiseLight else { return false }
        YFCommand.setRaiseLight(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置消息通知, 只有开关
    private func setMessageNotify(_ newInfo: YFMessageNotify, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.messageNotifyInfo else { return false }
        YFCommand.setMessageNotify(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置消息通知，应用开关
    private func setMessageNotifyType(_ newInfo: YFMessageNotifyType, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.remindNotifySwitch else { return false }
        YFCommand.setMessageNotifyType(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    
    /// : -
    /// 设置来电提醒，只有开关
    private func setRemindCall(_ newInfo: YFRemindCall, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.remindCall else { return false }
        YFCommand.setRemindCall(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置目标提醒，只有开关
    private func setRemindTarget(_ newInfo: YFRemindTarget, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.remindTarget else { return false }
        YFCommand.setRemindTarget(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置久坐提醒
    private func setRemindSedentar(_ newInfo: YFRemindSedentary, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.remindSedentary else { return false }
        YFCommand.setRemindSedentary(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
    /// 设置喝水提醒
    private func setRemindDrink(_ newInfo: YFRemindDrink, _ closure: @escaping YFWriteClosure) -> Bool {
        guard newInfo != YFDataManager.hybridInfo.remindDrink else { return false }
        YFCommand.setRemindDrink(newInfo).writeToDevice { error in
            if error == nil { YFDataManager.hybridInfo.update(newInfo) }
            closure(error)
        }
        return true
    }
}
