//
//  File.swift
//  CTFit
//
//  Created by Mac on 2020/9/25.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import UIKit

extension BDAncsAppType {
    var name: String {
        switch self {
        case .qq:       return "QQ"
        case .wechat:   return "WeChat"
        case .sina:     return "Sina"
        case .facebook: return "Facebook"
        case .line:     return "Line"
        case .instagram:return "Instagram"
        case .kakotalk: return "KaKoTalk"
        case .gmail:    return "Gmail"

        case .twitter:  return "Twitter"
        case .linkin:   return "Linkin"
        case .whatsapp: return "WhatsApp"
        case .skype:    return "Skype"
        case .sms:      return "SMS"
        case .email:    return "Email"
        }
    }

    var icon: UIImage? {
        switch self {
        case .qq:       return ImageRepo.App.qq
        case .wechat:   return ImageRepo.App.wechat
        case .sina:     return ImageRepo.App.sina
        case .facebook: return ImageRepo.App.facebook
        case .line:     return ImageRepo.App.line
        case .instagram:return nil
        case .kakotalk: return nil
        case .gmail:    return nil

        case .twitter:  return ImageRepo.App.twitter
        case .linkin:   return ImageRepo.App.linkin
        case .whatsapp: return ImageRepo.App.whatsapp
        case .skype:    return nil
        case .sms:      return ImageRepo.App.sms
        case .email:    return nil
        }
    }

    static var items: [CircleCheckerItem] = {  all.map { (e) -> CircleCheckerItem in CircleCheckerItem(e) } } ()
}


class CircleCheckerItem: NSObject {
    
    private(set) var ancsType: BDAncsAppType = .qq
    
    init(_ type: BDAncsAppType) {
        self.ancsType = type;
    }
    
    lazy var image: UIImage? = { ancsType.icon } ()

    lazy var grayImage: UIImage? = {
        if #available(iOS 10, *) {
            return image?.grayImage()
        } else {
            return image?.noir
        }
    }()
  
}

class CircleCheckerButton: UIButton {
    var item: CircleCheckerItem! {
        didSet {
            self.setImage(item.grayImage, for: .normal);
            self.setImage(item.image, for: .selected);

        }
    }
    func toggle() {
        self.isSelected = !isSelected;
    }
    
    func statusText() -> String {
        if self.isSelected {
            return String(format: "%@(%@)", item.ancsType.name, SRString.Device.notify_message_enable.locastr);
        } else {
            return String(format: "%@(%@)", item.ancsType.name, SRString.Device.notify_message_disable.locastr);
        }
    }
}

class CircleChecker: UIView {
    
    // MARK: - Properties
    let kRingWidth :CGFloat = 60;
    let kAnimDuration :Double = 0.3;
    
    /// Angle for each item in degree
    private var sweepAngle :CGFloat {
        return CGFloat(360) / CGFloat(BDAncsAppType.all.count);
    };
    
    /// Rotated
    private var startAngle :CGFloat = -90;
    private var circleCalculate :CircleCalculate = CircleCalculate.unit

    private lazy var centerLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        return label
    } ()
    
    private lazy var checkerButtons: [CircleCheckerButton] = {
        BDAncsAppType.items.map { (e) -> CircleCheckerButton in
            let button = CircleCheckerButton()
            button.item = e
            button.addTarget(self, action: #selector(onCheckItemClicked(sender:)), for: .touchUpInside);
            return button
        }
    }()
    
    var wihteList: [BDAncsAppType] = BDAncsAppType.all {
        didSet {
            for button in checkerButtons {
                button.isSelected = wihteList.contains(button.item.ancsType);
            }
        }
    }
    
    
    
    // MARK: - Initialize
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder);
        initialize();
    }
    
    override func layoutSubviews() {
        super.layoutSubviews();
        
        let size = self.bounds.size;
        
        circleCalculate.center.x = size.width / 2;
        circleCalculate.center.y = size.height / 2;
        circleCalculate.radius = size.width / 2 - kRingWidth / 2;
        
        // Layout Menus
        for index in 0..<checkerButtons.count {
            let button = checkerButtons[index];
            button.sizeToFit();
            let btnSize = button.intrinsicContentSize;
            let angle = startAngle + sweepAngle * CGFloat(index);
            let pos = circleCalculate.position(angle: angle);
            let ox = pos.x - btnSize.width / 2;
            let oy = pos.y - btnSize.height / 2;
            button.frame = CGRect(x: ox, y: oy, width: btnSize.width, height: btnSize.height);
        }
    }
    
    private func initialize() {
        // Set background image
        let backgroundImg = ImageRepo.HomeCircle.Menu.bacground
        self.layer.contents = backgroundImg.cgImage;
        self.layer.backgroundColor = UIColor.clear.cgColor;
        checkerButtons.forEach { (e) in addSubview(e) }
        
        addSubview(centerLabel)
        centerLabel.snp.makeConstraints { (make) in
            make.center.equalToSuperview();
        }
    }
    
    
    @objc private func onCheckItemClicked(sender: CircleCheckerButton) {
        sender.toggle();
        centerLabel.text = sender.statusText();
        if sender.isSelected {
            if !wihteList.contains(sender.item.ancsType) {
                wihteList.append(sender.item.ancsType)
            }
        } else {
            if let index = wihteList.firstIndex(of: sender.item.ancsType) {
                wihteList.remove(at: index)
            }
        }
    }
}
