//
//  CircleMenu.swift
//  CTFit
//
//  Created by Mac on 2019/6/21.
//  Copyright © 2019 shirajo. All rights reserved.
//

import UIKit

enum CircleMenuItemType: Int{
    case home = 0
    case sport
    case sleep
    case heartRate
    case settings
    case step
    case camera
    
    static var collection: [CircleMenuItemType] = [.home, .sport, .sleep, .heartRate, .settings, .step, .camera];
    
    var icon: UIImage? {
        switch self {
        case .home: return ImageRepo.HomeCircle.Menu.home
        case .sport: return ImageRepo.HomeCircle.Menu.gps
        case .sleep: return ImageRepo.HomeCircle.Menu.sleep
        case .heartRate: return ImageRepo.HomeCircle.Menu.heartrate
        case .settings: return ImageRepo.HomeCircle.Menu.settings
        case .step: return ImageRepo.HomeCircle.Menu.sport
        case .camera: return ImageRepo.HomeCircle.Menu.camera
        }
    }
}

class CircleMenuItem {
    var type: CircleMenuItemType = .home
    var image: UIImage? { type.icon }
    init(_ type: CircleMenuItemType = .home) {
        self.type = type;
    }
}

class CircleMenu: UIView {
    
    typealias DidSelectedMenu = (_ type: CircleMenuItemType, _ repeat: Bool) -> Void
    
    // MARK: - Properties
    fileprivate var didSelectedMenu: DidSelectedMenu?
    func onDidSelectedMenu(_ closure: @escaping DidSelectedMenu) {
        didSelectedMenu = closure
    }

    var currentItem :Int = 0;
    var items: [CircleMenuItem]! {
        didSet {
            loadMenuDatas();
        }
    }
    
    /// Center View
    var centerView: StepProgressBar!
    /// Completion percent for steps goal
    var progress: CGFloat = 0 {
        didSet{
            centerView.setProgress(progress, animated: false);
        }
    }
    
    
    fileprivate var backgroundImg: UIImage?;
    /// Menu buttons
    fileprivate var menuButtons = [UIButton]();
    /// Angle for each item in degree
    fileprivate var sweepAngle :CGFloat = 0;
    
    /// 环宽
    fileprivate let kRingWidth :CGFloat = 60;
    /// Duration
    fileprivate let kAnimDuration :Double = 0.3;
    /// 选中指示的角度(Top)
    fileprivate let kIndicatorAngle :CGFloat = 270;
    /// Rotated
    fileprivate var startAngle :CGFloat = -90;
    fileprivate var circleCalculate :CircleCalculate = CircleCalculate.unit;
    
    fileprivate var isBeingDragged = false;
    fileprivate var rotatedDegree: CGFloat = 0 {
        didSet {
            enforceAnglePos(angle: &self.rotatedDegree);
            setNeedsLayout();
        }
    }
    
    
    // MARK: - override
    override var intrinsicContentSize: CGSize {
        get {
            if backgroundImg != nil { return backgroundImg!.size; }
            return CGSize(width: 290, height: 290);
        }
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder);
        initialize();
    }
    
    // MARK: Overrides
    override func layoutSubviews() {
        super.layoutSubviews();
                
        let size = self.bounds.size;
        circleCalculate.center.x = size.width / 2;
        circleCalculate.center.y = size.height / 2;
        circleCalculate.radius = size.width / 2 - kRingWidth / 2;
        
        // Layout Center View
        if let centerV = centerView {
            let centerW = size.width - kRingWidth * 2;
            let centerH = size.height - kRingWidth * 2;
            let cox = size.width / 2 - centerW / 2;
            let coy = size.height / 2 - centerH / 2;
            centerV.frame = CGRect(x: cox, y: coy, width: centerW, height: centerH);
        }
        // Layout Menus
        for index in 0..<menuButtons.count {
            let button = menuButtons[index];
            button.sizeToFit();
            let btnSize = button.intrinsicContentSize;
            let angle = startAngle + rotatedDegree + sweepAngle * CGFloat(index);
            let pos = circleCalculate.position(angle: angle);
            let ox = pos.x - btnSize.width / 2;
            let oy = pos.y - btnSize.height / 2;
            button.frame = CGRect(x: ox, y: oy, width: btnSize.width, height: btnSize.height);
        }
    }
    
    fileprivate func initialize() {
        backgroundImg = ImageRepo.HomeCircle.Menu.bacground
        
        // Set background image
        self.layer.contents = backgroundImg?.cgImage;
        self.layer.backgroundColor = UIColor.clear.cgColor;
        invalidateIntrinsicContentSize();
        
        // Set Center View
        self.centerView = StepProgressBar();
        self.centerView?.backgroundColor = UIColor.clear;
        self.addSubview(self.centerView!);
    }
    
    /// Load menu item
    fileprivate func loadMenuDatas() {
        if let hasItems = items {
            menuButtons.removeAll();
            self.sweepAngle = CGFloat(360 / hasItems.count);
            for index in 0..<hasItems.count {
                let button = UIButton();
                button.tag = index;
                button.setImage(hasItems[index].image, for: .normal);
                button.addTarget(self, action: #selector(onMenuItemClicked(button:)), for:.touchUpInside);
                menuButtons.append(button);
                self.addSubview(button);
            }
        }
    }
    
    // MARK: - Actions
    
    @objc func onMenuItemClicked(button :UIButton){
        let index = button.tag;
        if index != currentItem {
            animateToSelectedItem(selected: index);
        } else {
            notifyItemSelected(selected: index, repeater: true);
        }
    }
    
    fileprivate func notifyItemSelected(selected :Int, repeater :Bool) {
        let item = self.items[selected];
        didSelectedMenu?(item.type, repeater);
    }
    
    // MARK: - Rotated
    
    func animateToSelectedItem(selected :Int) {
        if selected == currentItem {
            return;
        }
        currentItem = selected;
        // 计算距离选中位置的角度
        let anglePos = angleForItem(index: selected);
        let includedAngle = includedAngleWithIndicator(angle: anglePos);
        // Path Animate
        animateRotatedDegree(from: self.rotatedDegree, to: self.rotatedDegree + includedAngle);
    }
    
    
    
    
    // MARK: - Touch Event
    
    fileprivate var lastEventTime: TimeInterval = 0;
    fileprivate var degreeVelocity: CGFloat = 0;
    fileprivate let kMaxDegreeVelocity: CGFloat = 500;
    fileprivate var effectDegree: CGFloat = 0;
    
    fileprivate func startDragging(_ point: CGPoint) {
        if isBeingDragged {
            return;
        }
        let distance = circleCalculate.distanceToCenter(point: point);
        if distance <= (circleCalculate.radius + kRingWidth/2) && distance >= (circleCalculate.radius - kRingWidth/2) {
            isBeingDragged = true;
        }
    }
    
    override func touchesBegan(_ touches: Set<UITouch>, with event: UIEvent?) {
        cancelAllAnimation();
        super.touchesBegan(touches, with: event);
        let touch = touches.first;
        if let hasTouch = touch {
            lastEventTime = hasTouch.timestamp;
        }
        degreeVelocity = 0;
    }
    
    override func touchesMoved(_ touches: Set<UITouch>, with event: UIEvent?) {
        super.touchesMoved(touches, with: event);
        let touch = touches.first;
        if let hasTouch = touch {
            let location = hasTouch.location(in: self);
            startDragging(location);
            
            if isBeingDragged {
                let preLocation = hasTouch.previousLocation(in: self);
                let preAngle = circleCalculate.angle(forPoint: preLocation);
                let angle = circleCalculate.angle(forPoint: location);
                var offset = angle - preAngle;
                if abs(offset) > 180 {
                    offset = (360 - abs(offset)) * ( angle>0 ? -1 : 1)
                }
  
                rotate(angle: offset);
                let duration = hasTouch.timestamp - lastEventTime;
                degreeVelocity = offset / CGFloat(duration);
            }
            lastEventTime = hasTouch.timestamp;
        }
    }
    
    override func touchesEnded(_ touches: Set<UITouch>, with event: UIEvent?) {
        super.touchesEnded(touches, with: event);
        if abs(degreeVelocity) > 250 {
            if degreeVelocity > kMaxDegreeVelocity {
                degreeVelocity = kMaxDegreeVelocity;
            } else if degreeVelocity < -kMaxDegreeVelocity {
                degreeVelocity = -kMaxDegreeVelocity;
            }
            fling(with: degreeVelocity);
        } else {
            finishRotated();
        }
    }
    
    override func touchesCancelled(_ touches: Set<UITouch>, with event: UIEvent?) {
        super.touchesCancelled(touches, with: event);
        finishRotated();
    }
    
    fileprivate func rotate(angle: CGFloat) {
        effectDegree += angle;
        if abs(effectDegree) > 10 {
            effectDegree = 0;
            AudioManager.shared.playKaka();
        }
        self.rotatedDegree += angle;
        checkSelectedItem();
    }
    
    fileprivate func fling(with velocity: CGFloat) {
        let flingAnimator = self.flinganimator;
        if flingAnimator.hasStartedAnimating {
            flingAnimator.cancel();
        }
        flingAnimator.startValue = -velocity;
        flingAnimator.endValue = 0;
        flingAnimator.duration = calcFlingDuration(velocity: velocity);
        flingAnimator.onAnimationUpdate = {
            let offset = flingAnimator.animatedValue / 33;
            self.rotate(angle: -offset);
        }
        flingAnimator.onAnimationEnd = {
            if !flingAnimator.isCancelled {
                self.finishRotated();
            }
        }
        flingAnimator.start();
    }
    
    fileprivate lazy var flinganimator: AnimatorValueCGFloat = {
        return AnimatorValueCGFloat();
    }()
    
    fileprivate func calcFlingDuration(velocity: CGFloat) -> Double {
        var absVelocity = abs(velocity);
        var duration: Double = 0;
        while absVelocity > 30 {
            absVelocity /= 1.066;
            duration += 0.03;
        }
        return duration;
    }
    
    fileprivate func finishRotated() {
        let modulus = self.rotatedDegree.remainder(dividingBy: self.sweepAngle);
        let absModulus = abs(modulus);
        var endDegree = self.rotatedDegree;
        if self.rotatedDegree < 0 {
            if absModulus > sweepAngle/2 {
                endDegree += absModulus - sweepAngle;
            } else {
                endDegree -= modulus;
            }
        } else {
            if absModulus > sweepAngle/2 {
                endDegree += sweepAngle - absModulus;
            } else {
                endDegree -= modulus;
            }
        }
        isBeingDragged = false;
        animateRotatedDegree(from: rotatedDegree, to: endDegree);
    }
    
    fileprivate lazy var rotateAnimator: AnimatorValueCGFloat = {
        return AnimatorValueCGFloat();
    }()
    
    fileprivate func animateRotatedDegree(from start: CGFloat, to end: CGFloat) {
        let animator = self.rotateAnimator;//AnimatorValueCGFloat(startValue: start, endValue: end);
        if animator.hasStartedAnimating {
            animator.cancel();
        }
        animator.startValue = start;
        animator.endValue = end;
        animator.duration = 0.3;
        animator.onAnimationUpdate = {
            self.rotatedDegree = animator.animatedValue;
            self.setNeedsLayout();
        }
        animator.onAnimationEnd = {
            if !animator.isCancelled {
                self.notifyItemSelected(selected: self.currentItem, repeater: false);
            }
        }
        animator.start();
    }
    
    fileprivate func checkSelectedItem() {
        // -2, -1, 0, 1, 2
        for index in 0..<self.menuButtons.count {
            let itemDegree = angleForItem(index: index);
            let includedDegree = includedAngleWithIndicator(angle: itemDegree);
            if abs(includedDegree) < (sweepAngle/2) {
                currentItem = index;
                break;
            }
        }
    }
    
    fileprivate func cancelAllAnimation(){
        if self.rotateAnimator.hasStartedAnimating {
            self.rotateAnimator.cancel();
        }
        if self.flinganimator.hasStartedAnimating {
            self.flinganimator.cancel();
        }
    }
    
    // MARK: Convenience
    
    fileprivate func angleForItem(index :Int) -> CGFloat {
        return startAngle + CGFloat(index) * sweepAngle + rotatedDegree;
    }
    
    // 某点与Indicator位置的夹角
    func includedAngleWithIndicator(angle: CGFloat) -> CGFloat {
        var degree = angle;
        // 将角度范围限定为 0~360
        enforceAnglePos(angle: &degree);
        
        let offset = kIndicatorAngle - degree;
        if offset >= 0 && offset <= 180 {
            return offset;
        } else {
            if offset > 180 {
                return offset - 360;
            }
            return offset;
        }
    }
    
    func enforceAnglePos(angle :inout CGFloat) {
        while angle > 360 {
            angle -= 360;
        }
        while angle < 0 {
            angle += 360;
        }
    }
    
    func enforceRotatedAngle(angle :inout CGFloat) {
        while angle > 180 {
            angle -= 180;
        }
        while angle < -180 {
            angle += 180;
        }
    }
}
