//
//  CircleProgressBar.swift
//  CTFit
//
//  Created by Mac on 2019/6/21.
//  Copyright © 2019 shirajo. All rights reserved.
//

import UIKit

class CircleProgressBar: UIView {

    // MARK: Properties
    let kPI_2 = Double.pi / 2
    let kPi: CGFloat = CGFloat(Float.pi)
    let kAngleStart: CGFloat = CGFloat(Float.pi / 2)
    let kLineWidth :CGFloat = 20
    let kAnimDuration :Float = 1.0
    
    let progressLayer :CAShapeLayer = CAShapeLayer()
    let gradientLayer :CALayer      = CALayer()
    let leftGradientLayer   :CAGradientLayer  = CAGradientLayer()
    let rightGradientLayer  :CAGradientLayer  = CAGradientLayer()
    let topGradientLayer    : CAGradientLayer = CAGradientLayer()
    
    fileprivate var angleEnd: CGFloat = 0 {
        didSet {
            self.setNeedsDisplay()
        }
    }
    
    fileprivate var gradientEnable: Bool = true {
        didSet {
            if gradientEnable != oldValue {
                if gradientEnable {
                    self.gradientLayer.isHidden = false
                    self.gradientLayer.mask = self.progressLayer
                } else {
                    self.gradientLayer.isHidden = true
                    self.layer.addSublayer(self.progressLayer)
                    self.setNeedsDisplay()
                }
            }
        }
    }
    
    var progress: CGFloat = 0 {
        didSet {
            gradientEnable = !(progress > 1)
        }
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        initialize()
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        initialize()
    }
    
    override func sizeThatFits(_ size: CGSize) -> CGSize {
        super.sizeThatFits(size)
        let temp = min(size.width, size.height)
        return CGSize(width: temp, height: temp)
    }
    
    override func draw(_ rect: CGRect) {
        
        // Draw progress layer
        let width = self.frame.size.width
        let height = self.frame.size.height
        let center = CGPoint(x: width / 2, y: height / 2 )
        let radius = (width / 2 - kLineWidth / 2)
        let progressArcPath = UIBezierPath(arcCenter: center,
                                           radius: radius,
                                           startAngle: kAngleStart,
                                           endAngle: kAngleStart + kPi * 2,
                                           clockwise: true)
        self.progressLayer.path = progressArcPath.cgPath
        
        leftGradientLayer.frame = CGRect(x: 0, y: height/4, width: width / 2, height: height*3/4)
        rightGradientLayer.frame = CGRect(x: width / 2, y: height/4, width: width / 2, height: height*3/4)
        topGradientLayer.frame = CGRect(x: 0, y: 0, width: width, height: height/4)
        
        super.draw(rect)
    }
    
    internal func initialize() {
        self.layer.backgroundColor = UIColor.clear.cgColor
        
        // Init Progress Layer
        self.progressLayer.name = "ProgressLayer"
        self.progressLayer.lineWidth = kLineWidth
        // self.progressLayer.lineCap = kCALineJoinRound
        self.progressLayer.fillColor = UIColor.clear.cgColor
        self.progressLayer.strokeColor = UIColor.red.cgColor
        self.layer.addSublayer(self.progressLayer)
        
        
        /// Init Gradient Layers, 120 degress
        ///
        /// (0,0)--------------------(1,0)
        ///  |                                |
        ///  |                                |
        ///  |          0.5                 |
        ///  |          0.5                 |
        ///  |                                |
        ///  |                                |
        /// (0,1)--------------------(1,1)
        ///
        let green   = Colors.StepGoal.green
        let blue    = Colors.StepGoal.blue
        let yellow  = Colors.StepGoal.yellow
        let red     = Colors.StepGoal.red
        
        leftGradientLayer.colors = [green.cgColor, blue.cgColor]
        leftGradientLayer.locations = [0, 1]
        leftGradientLayer.startPoint = CGPoint(x: 0.5, y: 1)
        leftGradientLayer.endPoint = CGPoint(x: 0.5, y: 0) // 0.5, 1
        self.gradientLayer.addSublayer(leftGradientLayer)
        
        topGradientLayer.colors = [blue.cgColor, yellow.cgColor]
        topGradientLayer.locations = [0.5, 0.8]
        topGradientLayer.startPoint = CGPoint(x: 0, y: 0.5)
        topGradientLayer.endPoint = CGPoint(x: 1, y: 0.5)
        self.gradientLayer.addSublayer(topGradientLayer)
        
        rightGradientLayer.colors = [red.cgColor, yellow.cgColor]
        rightGradientLayer.locations = [0, 1]
        rightGradientLayer.startPoint = CGPoint(x: 0.5, y: 1)
        rightGradientLayer.endPoint = CGPoint(x: 1, y: 0.3) // 0.5, 1
        self.gradientLayer.addSublayer(rightGradientLayer)
        
        self.gradientLayer.mask = self.progressLayer
        self.layer.addSublayer(self.gradientLayer)
    }
    
    func setProgress(_ progress: CGFloat, animated: Bool) {
        var toProgress = progress
        if toProgress < 0 { toProgress = 0 }
        
        if animated {
            CATransaction.begin()
            let animation = CABasicAnimation(keyPath: "strokeEnd")
            animation.isRemovedOnCompletion = false
            animation.fillMode = CAMediaTimingFillMode.forwards
            animation.duration = 1
            animation.fromValue = self.progress
            animation.toValue = toProgress
            self.progressLayer.add(animation, forKey: "progress")
            //self.progressLayer.strokeEnd = toProgress
            self.progress = toProgress
            CATransaction.commit()
        } else {
            self.progressLayer.strokeEnd = toProgress
            self.progress = toProgress
            setNeedsDisplay()
        }
        
    }
}

class StepProgressBar: CircleProgressBar {
    private lazy var imageView: UIImageView = {
        UIImageView(image: ImageRepo.HomeCircle.steps_larger)
    }()
    private lazy var valueLabel :UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = UIFont.boldCustomFont(ofSize: 19)
        label.textColor = Colors.white
        label.text  = "0"
        return label
    } ()
    private lazy var percentLabel: UILabel  = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.white
        label.text  = "0%"
        return label
    } ()
    
    var value: String? {
        didSet{
            valueLabel.text = value
        }
    }
    
    override var progress: CGFloat {
        didSet{
            gradientEnable = !(progress > 1)
            percentLabel.text = String(format: "%d%%", Int(progress * 100))
        }
    }
    
    override func initialize() {
        super.initialize()

        let items = [valueLabel, imageView, percentLabel]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.center.equalToSuperview()
        }
        
        imageView.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.bottom.equalTo(valueLabel.snp.top).offset(-8)
        }
        percentLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(valueLabel.snp.bottom).offset(8)
        }
    }
}
