//
//  BaseViewController.swift
//  CTFit
//
//  Created by Mac on 2020/5/22.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import RxSwift
import RxCocoa

class BaseViewController: UIViewController {
    
    lazy var disposeBag = DisposeBag()
    
    lazy var mainScheduler = MainScheduler.instance
    
    lazy var rightBarIsConnectState: Bool = false
    
    lazy var indicator: IndicatorView = IndicatorView()
    
    lazy var active: Bool = false
    
    // MARK: - Initialize & override
    
    deinit {
        UILog.d("\(classForCoder)")
    }
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
        
    init() {
        super.init(nibName: nil, bundle: nil)
        modalPresentationStyle = .fullScreen
    }
    
    override var preferredStatusBarStyle: UIStatusBarStyle {
        return .default;
    }
    
    override func viewDidLoad() {
        // self.jz_navigationBarTintColor = Colors.black
        self.jz_navigationBarBackgroundAlpha = 1
        if self.navigationController != nil { initNavigationBar() }
        super.viewDidLoad()
        view.backgroundColor = Colors.Base.background
        view.clipsToBounds = true
        if rightBarIsConnectState { observeBandConnectState() }
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        active = true
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        active = false
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        super.viewDidDisappear(animated)
    }
   
    internal func initNavigationBar() {
        self.jz_navigationBarHidden = false;
        if rightBarIsConnectState {
            connectStateBar()
        } else {
            if self.navigationController?.viewControllers.count==1 { return }
            self.navigationItem.leftBarButtonItem = backBarItem();
        }
    }
    
    
    // MARK: - onDidChanged
    
    typealias DidChangedClosure = (_ model: Any?) -> Void
    
    fileprivate var didChangedClosure: DidChangedClosure? = nil    
    
    final func onDidChanged(model: Any? = nil) {
        if let closure = didChangedClosure { closure(model) }
    }
    
    internal func onDidChangedClosure(_ closure: @escaping DidChangedClosure) {
        didChangedClosure = closure
    }
}

// MARK: - BarButtonItem

extension BaseViewController {
    
    @objc internal func backBarItem() -> UIBarButtonItem { /// 返回
        let item = UIBarButtonItem(image: ImageRepo.Navi.back, style: .plain, target: self, action: #selector(onBack));
        item.imageInsets = UIEdgeInsets(top: 0, left:-20, bottom: 0, right: 20);
        return item;
    }
    
    @objc internal func onBack(){
        self.navigationController?.popViewController(animated: true);
    }
    
    @objc internal func cancelBarItem() -> UIBarButtonItem { /// 取消
        let item = UIBarButtonItem(title: SRString.Base.cancel.locastr, style: .plain, target: self, action: #selector(onCancel))
        return item;
    }
    
    @objc internal func onCancel(){
        self.dismiss(animated: true, completion: nil)
    }
    
    @objc internal func shareBarItem() -> UIBarButtonItem { /// 分享
        let item = UIBarButtonItem(image: ImageRepo.Navi.share, style: .plain, target: self, action: #selector(onShare(sender:)));
        return item;
    }
       
    @objc internal func onShare(sender: UIBarButtonItem){
        guard let image = screenSnapshot() else { return }
        self.presentActitvity(image: image, barButtonItem: sender)
    }
    
    @objc internal func addBarItem() -> UIBarButtonItem? { /// 闹钟添加
        let item = UIBarButtonItem(title: SRString.Base.add.locastr, style: .plain, target: self, action: #selector(onAdd(_:)));
        return item;
    }
       
    @objc internal func onAdd(_ sender: UIBarButtonItem){
        self.navigationController?.popViewController(animated: true);
    }
    
    @objc private func connectedBarItem() -> UIBarButtonItem? { /// 已连接
        let backBar = UIBarButtonItem(image: ImageRepo.Navi.connected, style: .plain, target: self, action: #selector(onConnectedSelector));
        return backBar;
    }
                  
    @objc private func onConnectedSelector(sender: UIBarButtonItem) {
        navigationController?.pushViewController(ConnectedViewController(), animated: true)
    }
    
    @objc private func disconnectedBarItem() -> UIBarButtonItem? { /// 已断开
        let backBar = UIBarButtonItem(image: ImageRepo.Navi.disconnected, style: .plain, target: self, action: #selector(onDisconnectedSelector));
        return backBar;
    }
    
    @objc private func onDisconnectedSelector(sender: UIBarButtonItem) {
        navigationController?.pushViewController(SearchedViewController(), animated: true)
    }
}

// MARK: - 监听手环连接状态

extension BaseViewController {
    
    @objc internal func onConnected() { connectStateBar() }

    @objc internal func onDisconnected() { connectStateBar() }
    
    @objc private func connectStateBar() {
        guard rightBarIsConnectState else { return }
        if BandConnect.isConnected {
            navigationItem.rightBarButtonItem = connectedBarItem()
        } else {
            navigationItem.rightBarButtonItem = disconnectedBarItem()
        }
    }
    
    internal func observeBandConnectState() {
        BandConnect.statusChangedOutput.observeOn(MainScheduler.instance).subscribe(onNext: { [weak self] state in
            guard let strong = self else { return }
            switch state {
            case .connected: strong.onConnected()
            case .disconnected: strong.onDisconnected()
            }
        }).disposed(by: disposeBag)
    }
}

// MARK: -  截屏分享

extension BaseViewController {
    
    @objc internal func screenSnapshot(save: Bool = false) -> UIImage? {
        guard let window = UIApplication.shared.keyWindow else { return nil }
        // 用下面这行而不是UIGraphicsBeginImageContext()，因为前者支持Retina
        UIGraphicsBeginImageContextWithOptions(window.bounds.size, false, 0.0)
        window.layer.render(in: UIGraphicsGetCurrentContext()!)
        let image = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        guard let hasImage = image else { return nil; }
        if save { UIImageWriteToSavedPhotosAlbum(hasImage, self, nil, nil) }
        //let data = UIImagePNGRepresentation(hasImage);
        return image
    }
    
    /// 原生分享 UIActivityViewController
    ///  Failed to start plugin; plugin is already activating
    private func presentActitvity( image: UIImage, sourceView: UIView? = nil, barButtonItem: UIBarButtonItem? = nil){
        let activityVC = VisualActivityViewController(image: image)
        activityVC.restorationIdentifier = "com.shirajo.ctfit.restorationIdentifier";
        activityVC.modalPresentationStyle = .overFullScreen;
        activityVC.previewImageSideLength = 160
        if UIDevice.idiom == .pad {
            let popPresenter = activityVC.popoverPresentationController;
            if let view = sourceView {
                popPresenter?.sourceView = view;
                popPresenter?.sourceRect = view.bounds;
            } else if let item = barButtonItem {
                popPresenter?.barButtonItem = item;
            }
        }
        let parentVC = Helper.rootVc;
        parentVC?.present(activityVC, animated: true, completion: { });
    }
}

// MARK: - 深色模式
extension BaseViewController {
    internal func userInterfaceStyle(dark: Bool = false) {
        dark ? view.userDarkMode() : view.userLightMode()
    }
    /// 色彩模式发生变化
    override func traitCollectionDidChange(_ previousTraitCollection: UITraitCollection?) {
        super.traitCollectionDidChange(previousTraitCollection)
        /// 每次模式改变的时候, 这里都会执行
    }
}

// MARK: - IndicatorView
extension BaseViewController {
    internal func indicatorShow(message: String) {
        self.indicator.show(title: message, in: navigationItem, rootUserInteractionEnabled: false);
    }
    internal func indicatorDismiss() {
        self.indicator.dismiss();
    }
}
