//
//  GuidView.swift
//  CTFit
//
//  Created by Mac on 2021/1/29.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import UIKit
import Foundation
import SnapKit
import RxSwift

class GuidView: UIView {
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        initialize()
    }
    
    private func initialize() {
        backgroundColor = Colors.rgb(0x3b3b3b, 0xb3/0xFF)
        addSubviews()
    }
        
    private func addSubviews() {
        [arrowView, describeLabel, bandButton].forEach { e in
            e.translatesAutoresizingMaskIntoConstraints = false
            addSubview(e)
        }

        arrowView.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(UIDevice.topLayoutGuide+34);
            make.right.equalToSuperview().offset(-8);
        }
        
        describeLabel.snp.makeConstraints { (make) in
            make.top.equalTo(arrowView.snp.bottom).offset(16);
            make.left.equalToSuperview().offset(16);
            make.right.equalToSuperview().offset(-16);
        }
        
        bandButton.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(UIDevice.topLayoutGuide)
            make.right.equalToSuperview();
            make.height.equalTo(44);
            make.width.equalTo(88);
        }
    }
    
    private var arrowView: UIImageView = {
        let view = UIImageView(image: ImageRepo.Other.guide_arrow)
        return view
    } ()
    
    private var describeLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.white
        label.font = .customFont(ofSize: 28)
        label.numberOfLines = 0
        label.sizeToFit()
        label.text = SRString.Base.guide_pair_message.locastr
        return label
    } ()
    
    private var bandButton: UIButton = {
        let button = UIButton()
        button.addTarget(self, action: #selector(onClickedBand), for: .touchUpInside)
        return button
    } ()
    
    @objc
    private func onClickedBand() {
        GuidView.hidden()
        onSearchBand()
    }
    
    private func onSearchBand() {
        guard let currentVc = Helper.currentVc else { return }
        guard currentVc.isKind(of: HomeCircleContrller.self) || currentVc.isKind(of: HomeListController.self) else { return }
        currentVc.navigationController?.pushViewController(SearchedViewController(), animated: true)
    }
}

extension GuidView {
    private static let key = "kGuidePairConnected"
    public static var valid: Bool {
        get {
            if let _ = UserDefaults.standard.value(forKey: key) {
                return UserDefaults.standard.bool(forKey: key)
            } else {
                return true
            }
        }
        set {
            UserDefaults.standard.set(newValue, forKey: key)
            UserDefaults.standard.synchronize()
        }
    }
}

extension GuidView {
    
    private static var guidView: GuidView?
    
    private static func hidden() {
        guard guidView?.superview != nil else { return }
        guidView?.removeFromSuperview();
        guidView = nil
    }
    
    static func show() {
        guard valid, let window = Helper.keyWindow, guidView == nil else { return }
        valid = false
        guidView = GuidView()
        [guidView!].forEach { e in
            e.translatesAutoresizingMaskIntoConstraints = false
            window.addSubview(e)
        }
        guidView!.snp.makeConstraints { (make) in
            make.left.top.right.bottom.equalToSuperview();
        }
    }
}

extension GuidView {
    private static var disposeBag = DisposeBag()

    private static func observeBandConnectState(){
        BandConnect.statusChangedOutput.observeOn(MainScheduler.instance).subscribe(onNext: { state in
            if case .connected = state { hidden() }
        }).disposed(by: disposeBag)
    }
    
    static func loading() {
        guard valid, !BandConnect.isConnected else { return }
        show()
        observeBandConnectState()

    }
}
