//
//  IndicatorView.swift
//  CTFit
//
//  Created by Mac on 2020/9/7.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit

class IndicatorView: UIView {
    var text: String?
    var showing: Bool = false
    fileprivate var navigationItem: UINavigationItem?
    fileprivate var leftBarItem: UIBarButtonItem?
    fileprivate var rightBarItem: UIBarButtonItem?
    fileprivate var rootUserInteractionEnabled: Bool = true {
        didSet {
            Helper.keyWindow?.isUserInteractionEnabled = rootUserInteractionEnabled
            guard navigationItem != nil else { return }
            if !rootUserInteractionEnabled {
                leftBarItem = navigationItem?.leftBarButtonItem
                rightBarItem = navigationItem?.rightBarButtonItem
                navigationItem?.leftBarButtonItem = nil
                navigationItem?.rightBarButtonItem = nil
                navigationItem?.hidesBackButton = true
            } else {
                navigationItem?.leftBarButtonItem = leftBarItem
                navigationItem?.rightBarButtonItem = rightBarItem
            }
        }
    }
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: CGRect(x: 0, y: 0, width: UIDevice.scrWidth, height: UIDevice.naviHeight))
        addSubviews()
    }
    
    
    // MARK: - View customer
    
    fileprivate func addSubviews() {
        [titleLabel, indicator].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        titleLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.centerX.equalToSuperview().offset(20)
        }
        
        indicator.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.right.equalTo(titleLabel.snp.left).offset(-4)
        }
        
    }
    
    fileprivate lazy var indicator: UIActivityIndicatorView = {
        let view = UIActivityIndicatorView()
        return view
    } ()
    
    fileprivate lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.naviTitle)
        return label
    } ()
}

extension IndicatorView {
    public func show(title: String?, in navigationItem: UINavigationItem, rootUserInteractionEnabled: Bool = true) {
        guard !showing else { return }
        showing = true
        self.titleLabel.text = title
        self.navigationItem = navigationItem
        self.indicator.startAnimating()
        self.rootUserInteractionEnabled = rootUserInteractionEnabled
        navigationItem.titleView = self
        DispatchQueue.main.asyncAfter(deadline: .now()+30) {
            self.dismiss()
        }
    }
    
    public func dismiss() {
        guard showing else { return }
        guard navigationItem != nil else { return }
        self.rootUserInteractionEnabled = true
        self.navigationItem?.titleView = nil
        self.indicator.stopAnimating()
        showing = false
    }
}
