//
//  CalendarPicker.swift
//  CTFit
//
//  Created by Mac on 2020/6/2.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import FSCalendar
import RxSwift

class CalendarPicker: UIView {
    
    typealias DidPickerClosure = (CalendarPicker) -> Void

    var didPicker: DidPickerClosure?
    
    var minDate :Date = CalendarUtils.date(from: "2010-01-01", format: "yyyy-MM-dd") { didSet { calendarView.reloadData() } }
    
    var maxDate: Date = Date() { didSet { calendarView.reloadData() }  }
    
    var selectedDate: Date {
        get { return calendarView.selectedDate! }
        set { calendarView.select(newValue) }//选择日期
    }
    
    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    init(with date: Date = Date(), didPicker: DidPickerClosure? = nil) {
        super.init(frame: .zero)
        backgroundColor = Colors.Cell.background
        self.didPicker = didPicker
        addSubviews()
        setupConstraints()
        self.selectedDate = date
        setUpContentButtonTitle()
        observeAppNotify()
    }

    
    // MAKR: - internal
    
    private let disposeBag = DisposeBag()
    
    let calendar = CalendarUtils.calendar
    
    lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.title)
        label.font = .lightCustomFont(ofSize: 20)
        label.textAlignment = .center
        label.textColor = Colors.blue
        return label
    }()
    
    lazy var leftButton: UIButton = {
        let button = UIButton(type: .system)
        button.tintColor = Colors.Base.foreground
        button.setImage(ImageRepo.Other.previous, for: .normal)
        button.addTarget(self, action: #selector(onLeftButtonTap(_:)), for: .touchUpInside)
        return button
    }()
    
    lazy var rightButton: UIButton = {
        let button = UIButton(type: .system)
        button.tintColor = Colors.Base.foreground
        button.setImage(ImageRepo.Other.next, for: .normal)
        button.addTarget(self, action: #selector(onRightButtonTap(_:)), for: .touchUpInside)
        return button
    }()
    
    lazy var calendarView: FSCalendar = {
        let _calendarView = FSCalendar();
        _calendarView.dataSource = self;
        _calendarView.delegate = self;
        
        /// 不显示默认的头部视图
        _calendarView.headerHeight = 0
        /// 设置周显示高度为 40
        _calendarView.weekdayHeight = 30
        
        _calendarView.fs_height = 250
        
        _calendarView.rowHeight = 40
        
        /// 默认选择日期是：今天
        _calendarView.select(Date());
        
        /// 设置周次显示为： 一 、二 .
        _calendarView.appearance.caseOptions = .weekdayUsesSingleUpperCase
        /// 设置周(日、一、二、三、四) 字体颜色
        _calendarView.appearance.weekdayTextColor = Colors.blue
        /// 设置周六、周日下文字颜色
        /// _calendarView.appearance.titleWeekendColor = Colors.blue
        _calendarView.placeholderType = .fillHeadTail
        /// 非本月 Day 字体颜色
        _calendarView.appearance.titlePlaceholderColor = Colors.gray
        /// 设置默认日期字体颜色
        _calendarView.appearance.titleDefaultColor = Colors.Base.foreground
        /// 设置选中日期字体颜色
        _calendarView.appearance.titleSelectionColor = Colors.blue
        /// 设置今天文字颜色
        _calendarView.appearance.titleTodayColor = Colors.blue
        /// 设置选中后背景颜色
        _calendarView.appearance.selectionColor = .clear
        /// 设置单元格圆角
        /// 1.0=圆，0.0=正方形，0.0-1.0之间的值为圆角
        _calendarView.appearance.borderRadius = 0.4;
        /// 设置翻页方式为水平
        _calendarView.scrollDirection = .horizontal
        /// 设置是否用户多选
        _calendarView.allowsMultipleSelection = false
        /// 不默认标志今天
        _calendarView.today = nil; // Don't use default specified
        
        return _calendarView
    }()
    
    private func observeAppNotify() {
        BandData.appNotifyOutput.subscribe(onNext: { [weak self] type in
            guard let strong = self else { return }
            switch type {
            case .significantTimeChange: strong.onSignificantTimeChange()
            default: break
            }
        }).disposed(by: disposeBag)
    }
    
    private func onSignificantTimeChange() {
        calendarView.reloadData()
        self.delayUpdate()
    }
    
    private func delayUpdate() {
        let now = Date()
        maxDate = now
        guard let selectDate = calendarView.selectedDate else { return }
        let formDate = CalendarUtils.dayDate(for: selectDate)
        let toDate = CalendarUtils.dayDate(for: now)
        let days = CalendarUtils.componentDay(from: formDate, to: toDate)
        if days != 0 { calendarView.select(now) }
    }
    
    
    private func addSubviews() {
        [titleLabel, calendarView, leftButton, rightButton].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

    }

    private func setupConstraints() {
        titleLabel.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(UIDevice.topLayoutGuide)
            make.width.equalTo(UIDevice.scrWidth)
            make.height.equalTo(40)
        }
        
        calendarView.snp.makeConstraints { (make) in
            make.top.equalTo(titleLabel.snp.bottom).offset(Constraints.small)
            make.left.right.equalToSuperview()
            make.height.equalTo(calendarView.fs_height)
            make.bottom.equalToSuperview().offset(-4)
        }
        
        leftButton.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(-16)
            make.centerY.equalTo(titleLabel.snp.centerY)
            make.width.greaterThanOrEqualTo(80)
        }
        
        rightButton.snp.makeConstraints { (make) in
            make.right.equalToSuperview().offset(16)
            make.centerY.equalTo(titleLabel.snp.centerY)
            make.width.greaterThanOrEqualTo(80)
        }
        
        let height = UIDevice.topLayoutGuide + 40 + calendarView.fs_height
        self.bounds = CGRect(x: 0, y: 0, width: UIDevice.scrWidth, height: height)
    }
}

extension CalendarPicker {
    
    @objc
    private func onLeftButtonTap(_ sender: UIButton) {
        guard let previousMonth = calendar.date(byAdding: .month, value: -1, to: calendarView.currentPage) else { return }
        calendarView.setCurrentPage(previousMonth, animated: true)
    }
    
    @objc
    private func onRightButtonTap(_ sender: UIButton) {
        guard let nextMonth = calendar.date(byAdding: .month, value: 1, to: calendarView.currentPage) else { return }
        calendarView.setCurrentPage(nextMonth, animated: true)
    }
    
    
    private func setUpContentButtonTitle() {
        let date = calendarView.currentPage
        let unitFlags = CalendarUtils.kDateComponentAll
        let components =  calendar.dateComponents(unitFlags, from: date)
        let year    = UInt16(components.year!)
        let month   = UInt8(components.month!)
        self.titleLabel.text = String(format: "%04d-%02d", year, month)
        
        updateArrowVisible()
    }
    
    private func updateArrowVisible() {
        //leftButton.isHidden = !hasPrePage();
        //rightButton.isHidden = !hasNextPage();
        leftButton.isEnabled = hasPrePage();
        rightButton.isEnabled = hasNextPage();
    }
}


extension CalendarPicker: FSCalendarDataSource, FSCalendarDelegate {
    
    func minimumDate(for calendar: FSCalendar) -> Date {
        return minDate;
    }
       
    func maximumDate(for calendar: FSCalendar) -> Date {
        return maxDate;
    }
    
    func calendar(_ calendar: FSCalendar, didSelect date: Date, at monthPosition: FSCalendarMonthPosition) {
        UILog.v("didSelect date: \(date), monthPosition:\(monthPosition)")
        didPicker?(self)
    }
    
    func calendarCurrentPageDidChange(_ calendar: FSCalendar) {
        setUpContentButtonTitle()
    }
    
}

extension CalendarPicker {
    
    func hasPrePage()-> Bool {
        if calendar.isSame(in: .year, date: calendarView.currentPage, to: minDate),
            calendar.isSame(in: .month, date: calendarView.currentPage, to: minDate){
            return false
        }
        return true
    }
    
    func hasNextPage() -> Bool {
        if calendar.isSame(in: .year, date: calendarView.currentPage, to: maxDate),
            calendar.isSame(in: .month, date: calendarView.currentPage, to: maxDate){
            return false
        }
        return true
    }
    
}
