//
//  DateBar.swift
//  CTFit
//
//  Created by Mac on 2020/6/2.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import RxSwift

enum IntervalStyle {
    case day
    case week
    case month
    case year
    
    var perUnit: String {
        switch self {
        case .day: return "/\(SRString.Unit.hour.locastr)"
        case .week, .month: return "/\(SRString.Base.day.locastr)"
        case .year: return "/\(SRString.Base.month.locastr)"
        }
    }
    
    var stepGroundImage: UIImage {
        switch self {
        case .day: return ImageRepo.Step.bg_day
        case .week: return ImageRepo.Step.bg_week
        case .month: return ImageRepo.Step.bg_month
        case .year: return ImageRepo.Step.bg_year
        }
    }
    
    var sleepGroundImage: UIImage {
        switch self {
        case .day: return ImageRepo.Sleep.bg_day
        case .week: return ImageRepo.Sleep.bg_week
        case .month: return ImageRepo.Sleep.bg_month
        case .year: return ImageRepo.Sleep.bg_year
        }
    }
}

extension IntervalStyle {
    
    private static var formatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "yyyy-MM-dd";
        return fm;
    } ()
    
    static func timePeridDay(_ date: Date) -> String {
        let formatter = IntervalStyle.formatter
        return formatter.string(from: date)
    }
    
    static func timePeridWeek(_ weekInterval: DateInterval) -> String {
        let formatter = IntervalStyle.formatter
        let calendar = CalendarUtils.calendar
        
        //// 三种情况: 1、同年同月, 2、同年不同月, 3、不同年
        let sameYear = calendar.isSame(in: .year, date: weekInterval.start, to: weekInterval.end);
        let sameMonth = calendar.isSame(in: .month, date: weekInterval.start, to: weekInterval.end);
        
        var startText: String!;
        var endText: String!;
        if sameYear && sameMonth {
            formatter.dateFormat = "MMM d";
            startText = formatter.string(from: weekInterval.start);
            formatter.dateFormat = "d"
            endText = formatter.string(from: weekInterval.end);
        } else if sameYear && !sameMonth {
            formatter.dateFormat = "MMM d";
            startText = formatter.string(from: weekInterval.start);
            endText = formatter.string(from: weekInterval.end);
        } else {
            formatter.dateFormat = "yyyy MMM d";
            startText = formatter.string(from: weekInterval.start);
            endText = formatter.string(from: weekInterval.end);
        }
        return String(format: "%@-%@", startText, endText);
    }
    
    static func timePeridMonth(_ monthInterval: DateInterval) -> String {
        let formatter = IntervalStyle.formatter
        let calendar = CalendarUtils.calendar
        
        let isSameYear = calendar.isSame(in: .year, date: monthInterval.start, to: monthInterval.end);
        if isSameYear {
            formatter.dateFormat = "MMM";
        } else {
            formatter.dateFormat = "yyyy-MMM";
        }
        return formatter.string(from: monthInterval.start);
    }
    
    static func timePeridYear(_ yearInterval: DateInterval) -> String {
        let formatter = IntervalStyle.formatter
        formatter.dateFormat = "yyyy";
        return formatter.string(from: yearInterval.start)
    }
}

class DateBar: UIControl {
    
    var intervalStyle :IntervalStyle = .day {
        didSet {
            if intervalStyle != oldValue {
                self.initialize();
            }
        }
    }
    
    private lazy var dateBarDelegate :DateBarDelegate = {
        switch intervalStyle {
        case .day:
            return DayDateBarDelegate();
        case .week:
            return WeekDateBarDelegate();
        case .month:
            return MonthDateBarDelegate();
        case .year:
            return YearDateBarDelegate();
        }
    } ()

    
    var selectedDate: Date? {
        guard self.intervalStyle == .day,
              let dayDateBar = dateBarDelegate as? DayDateBarDelegate else { return nil }
        return dayDateBar.date;
    }
    
    var selectedWeekend: DateInterval? {
        guard self.intervalStyle == .week,
              let weekDateBar = dateBarDelegate as? WeekDateBarDelegate else { return nil }
        return weekDateBar.weekInterval;
    }
    
    var selectedMonth: DateInterval? {
        guard self.intervalStyle == .month,
              let monthDateBar = dateBarDelegate as? MonthDateBarDelegate else { return nil }
        return monthDateBar.monthInterval;
    }
    
    var selectedYear: DateInterval? {
        guard self.intervalStyle == .year,
              let yearDateBar = dateBarDelegate as? YearDateBarDelegate else { return nil }
        return yearDateBar.yearInterval;
    }
    
    var selectedDateInterval: DateInterval? {
        switch intervalStyle {
        case .day: return nil
        case .week: return selectedWeekend
        case .month: return selectedMonth
        case .year: return selectedYear
        }
    }
    
    var title: String {
        get { return dateBarDelegate.titleText() }
    }
    
    /// 判断是否选中了今天(本周/本月/今年)
    var isTodaySelected: Bool {
        return dateBarDelegate.isTodaySelected();
    }
    
    weak var attachedViewController: UIViewController?;
    
    private var titleButton :UIButton = {
        let button = UIButton(type: .system)
        button.tintColor = Colors.Base.foreground
        button.titleLabel?.font = .lightCustomFont(ofSize: 20)
        button.addTarget(self, action: #selector(titleTapped(sender:)), for: .touchUpInside)
        return button
    }()
    
    private lazy var previousButton: UIButton = {
        let button = UIButton(type: .system)
        button.tintColor = Colors.Base.foreground
        button.setImage(ImageRepo.Other.previous, for: .normal)
        button.addTarget(self, action: #selector(preAction(sender:)), for: .touchUpInside)
        return button
    }()
    
    private var nextButton :UIButton = {
        let button = UIButton(type: .system)
        button.tintColor = Colors.Base.foreground
        button.setImage(ImageRepo.Other.next, for: .normal)
        button.addTarget(self, action: #selector(nextAction(sender:)), for: .touchUpInside)
        return button
    }()
    
    private let disposeBag = DisposeBag()

    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
       
    private init() {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }
    
    convenience init(style: IntervalStyle = .day) {
        self.init()
        self.intervalStyle = style
        initialize()
        observeAppNotify()
    }
    
    
    func loadWithDate(_ date:Date) {
        if self.intervalStyle == .day {
             (dateBarDelegate as! DayDateBarDelegate).date = date;
        } else if self.intervalStyle == .week {
            (dateBarDelegate as! WeekDateBarDelegate).setWeekInterval(date);
        } else if self.intervalStyle == .month {
            (dateBarDelegate as! MonthDateBarDelegate).setMonthInterval(date);
        } else if self.intervalStyle == .year {
            (dateBarDelegate as! YearDateBarDelegate).setYearInterval(date);
        }
        updateDisplay();
    }
    
    
    private func addSubviews() {
        [previousButton, nextButton, titleButton].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        titleButton.snp.makeConstraints { (make) in
            make.top.bottom.centerX.equalToSuperview()
            make.width.greaterThanOrEqualTo(60)
        }
        
        previousButton.snp.makeConstraints { (make) in
            make.top.bottom.left.equalToSuperview()
            make.width.greaterThanOrEqualTo(120)
        }
        
        nextButton.snp.makeConstraints { (make) in
            make.top.bottom.right.equalToSuperview()
            make.width.greaterThanOrEqualTo(120)
        }
    }

    
    private func initialize(){
        dateBarDelegate = createDateBarDelegate();
        dateBarDelegate.minDate = CalendarUtils.date(from: "2015-01-01", format: "yyyy-MM-dd");
        dateBarDelegate.maxDate = Date();

        dateBarDelegate.initialize();
        updateDisplay();
        dateBarDelegate.callback = {
            self.notifyDateChanged();
        }
    }
    
    private func observeAppNotify() {
        BandData.appNotifyOutput.subscribe(onNext: { [weak self] type in
            guard let strong = self else { return }
            switch type {
            case .significantTimeChange: strong.onSignificantTimeChange()
            default: break
            }
        }).disposed(by: disposeBag)
    }
    
    private var lastSignificantChangeDate: Date = Date()

    private func onSignificantTimeChange() {
        let now = Date()
        guard !CalendarUtils.inSameDay(now, lastSignificantChangeDate) else { return }
        lastSignificantChangeDate = now
        switch intervalStyle {
        case .day:
            guard let selectDate = selectedDate,
                  !CalendarUtils.inSameDay(now, selectDate) else {  return }
            dateBarDelegate.maxDate = now;
            loadWithDate(now)
        case .week, .month, .year:
            guard let dateInterval = selectedDateInterval,
                  !CalendarUtils.inSameDay(now, dateInterval.createDate) else { return }
            loadWithDate(now)
            notifyDateChanged()
        }
    }
    
    private func createDateBarDelegate() -> DateBarDelegate {
        switch intervalStyle {
        case .day:
            return DayDateBarDelegate();
        case .week:
            return WeekDateBarDelegate();
        case .month:
            return MonthDateBarDelegate();
        case .year:
            return YearDateBarDelegate();
        }
    }
    
    private func titleText() -> String {
        switch intervalStyle {
        case .day: return IntervalStyle.timePeridDay(selectedDate!)
        case .week: return IntervalStyle.timePeridWeek(selectedWeekend!);
        case .month: return IntervalStyle.timePeridMonth(selectedMonth!);
        case .year: return IntervalStyle.timePeridYear(selectedYear!);
        }
    }
    
    func updateDisplay() {
        setTitle(dateBarDelegate.titleText());
        previousButton.isEnabled = dateBarDelegate.shouldShowPreButton();
        nextButton.isEnabled = dateBarDelegate.shouldShowNextButton();
    }
}


// MARK:  - 控件方法

extension DateBar {
    
    func setTitle(_ title :String) {
        self.titleButton.titleLabel?.text = title;
        self.titleButton.setTitle(title, for: .normal);
    }
    
    @objc fileprivate func titleTapped(sender: UIButton) {
        dateBarDelegate.onTitleClicked(viewController: self.attachedViewController);
    }
    //前一天
    @objc fileprivate func preAction(sender :Any){
        if dateBarDelegate.onPreAction() {
            notifyDateChanged();
        }
    }
    //后一天
    @objc fileprivate func nextAction(sender :Any){
        if dateBarDelegate.onNextAction() {
            notifyDateChanged();
        }
    }
    
    fileprivate func notifyDateChanged() {
        updateDisplay();
        sendActions(for: .valueChanged);
    }
}
