//
//  DateInterval.swift
//  CTFit
//
//  Created by Mac on 2020/6/2.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation

struct DateInterval {
    
    /// 开始时间
    private(set) var start: Date
    /// 结束时间 （ > start ）
    private(set) var end: Date
    
    private(set) var createDate: Date

    
    /// 真实结束时间
    var realEnd: Date {
        return dateForStart(day: daysPassed-1) 
    }
    
    /// 区间内总天数
    var daysInterval: Int { max(CalendarUtils.componentDay(from: start, to: end) + 1, 0) }
    /// 区间内已过去的天数：包括今天
    var daysPassed: Int {
        let days = max(CalendarUtils.componentDay(from: start, to: Date()) + 1, 0)
        return days > daysInterval ? daysInterval : days
    }
    
    /// 区间内总月数
    var monthsInterval: Int { max(CalendarUtils.componentMonth(from: start, to: end) + 1, 0) }
    /// 区间内已过去的月数：包括本月
    var monthPassed: Int {
        let months = max(CalendarUtils.componentMonth(from: start, to: Date()) + 1, 0)
        return months > monthsInterval ? monthsInterval : months
    }
    
    init(start: Date, end: Date) {
        self.start = start;
        self.end = end;
        createDate = Date()
    }
    
    func dateForStart(day offset: Int) -> Date {
        CalendarUtils.calendar.date(byAdding: .day, value: offset, to: start) ?? start;
    }
    
    func dateForStart(month offset: Int) -> Date {
        CalendarUtils.calendar.date(byAdding: .month, value: offset, to: start) ?? start;
    }
    
    func contains(_ date: Date) -> Bool {
        if date.timeIntervalSince1970 >= start.timeIntervalSince1970,
           date.timeIntervalSince1970 <= realEnd.timeIntervalSince1970 { return true }
        return false
    }
}

extension DateInterval: CustomStringConvertible {
    var description: String {
        return "(\(start) ~ \(end)), days: \(daysInterval), daysPassed:\(daysPassed)";
    }
}
