//
//  File.swift
//  CTFit
//
//  Created by Mac on 2020/5/26.
//  Copyright © 2020 shirajo. All rights reserved.
//

// -setNeedsLayout方法：标记为需要重新布局，异步调用layoutIfNeeded刷新布局，不立即刷新，但layoutSubviews一定会被调用
// -layoutIfNeeded方法：如果有需要刷新的标记，立即调用layoutSubviews进行布局（如果没有标记，不会调用layoutSubviews）关于在什么的情景会使用该方法
// 如果要立即刷新，要先调用[view setNeedsLayout]，把标记设为需要布局，然后马上调用[view layoutIfNeeded]，实现布局
// 在视图第一次显示之前，标记肯定是“需要刷新”的，所以直接调用[view layoutIfNeeded]就会进行立即更新

// 告诉self.view约束需要更新
// self.needsUpdateConstraints()
// 调用此方法告诉self.view检测是否需要更新约束，若需要则更新，下面添加动画效果才起作用
// self.updateConstraintsIfNeeded()

import Foundation
import UIKit

enum shadowViewStyle {
    case normal
    case top
    case bottom
}

class Dialog: UIView {
    
    typealias CompleteClosure = (UIView?)->Void
    
    var style: shadowViewStyle = .normal {
        didSet {
            if style == .normal {
                shadowRadius = 8
            } else {
                shadowRadius = 8
            }
        }
    }
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(with style: shadowViewStyle = .bottom) {
        super.init(frame: UIDevice.scrBounds)
        resetInitialize()
        self.style = style
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        shadow(with: shadowView)
    }
    
    func pop(_ view: UIView, completed closure: @escaping CompleteClosure, dismissClosure: CompleteClosure? = nil) {
        if isShow { return }
        self.dismissClosure = dismissClosure
        isShow = true
        customView = view
        shadow(with: shadowView)
        shadowView.addSubview(view)
        view.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
        
        addSubview(shadowView)
        popAnimate { _ in closure(self.customView) }
    }
    
    func dismiss(_ animated: Bool, completed closure: @escaping CompleteClosure) {
        if !isShow { return }
        isShow = false
        if animated {
            dismissAnimate { _ in
                closure(self.customView)
                self.customView?.removeFromSuperview()
                self.removeFromSuperview()
            }
        } else {
            closure(self.customView)
            self.customView?.removeFromSuperview()
            self.removeFromSuperview()
        }
    }
    
    override func touchesBegan(_ touches: Set<UITouch>, with event: UIEvent?) {
        guard let touch: UITouch = event?.touches(for: self)?.first else { return }
        let location = touch.location(in: self)
        if location.x != 0, location.y != 0 {
            dismiss(true) { [weak self] _ in
                guard let strong = self else { return }
                guard let closure = strong.dismissClosure else { return }
                closure(strong.customView)
            }
        }
    }
    
    // MARK: - private
    
    private var isShow: Bool = false

    private var duration: TimeInterval = 0.35
    
    private var backgounedAlpha: CGFloat = 0.45
    
    private var customView: UIView?
    
    private var dismissClosure: CompleteClosure?

    private lazy var shadowView: UIView = {
        let view = UIView()
        view.layer.rasterizationScale = UIScreen.main.scale
        view.backgroundColor = .clear
        return view
    } ()
    
    private var displayShadow: Bool = false
    
    private var shadowColor: UIColor = Colors.darkMode(Colors.gray, Colors.dark)
    
    private var shadowOpacity: Float = 0.8

    public var shadowRadius: CGFloat = 12.0
  
    public var shadowOffset = CGSize(width: 0.0, height: 1.0)

    private func resetInitialize() {
        style = .bottom
        duration = 0.25
        backgounedAlpha = 0.35
        self.isUserInteractionEnabled = true
        self.layer.rasterizationScale = UIScreen.main.scale
    }
    
    private func shadow(with view: UIView) {
        if displayShadow {
            view.layer.shadowColor = shadowColor.cgColor
            view.layer.shadowOpacity = shadowOpacity
            view.layer.shadowRadius = shadowRadius
            view.layer.shadowOffset = shadowOffset
        }
    }
    
    let coverColor: UIColor = Colors.darkMode(Colors.black, Colors.black)

    private func popAnimate(completed closure: @escaping CompleteClosure) {
        guard let rootWindow = UIApplication.shared.keyWindow else { return }
        rootWindow.addSubview(self)
        self.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
        
        self.backgroundColor = self.coverColor.withAlphaComponent(0.0)
        /// 初始位置
        switch style {
        case .normal:
            shadowView.alpha = 0.0
            shadowView.snp.remakeConstraints { (make) in
                make.centerX.equalToSuperview()
                make.centerY.equalToSuperview()
            }
        case .top:
            shadowView.snp.remakeConstraints { (make) in
                make.centerX.equalToSuperview()
                make.bottom.equalTo(self.snp.top)
            }
        case .bottom:
            shadowView.snp.remakeConstraints { (make) in
                make.centerX.equalToSuperview()
                make.top.equalTo(self.snp.bottom)
            }
        }
        self.layoutIfNeeded()

        
        /// 更新动画
        switch style {
        case .normal: 
            UIView.animate(withDuration: duration, animations: { [unowned self] in
                self.shadowView.alpha = 1.0
                self.backgroundColor = self.coverColor.withAlphaComponent(self.backgounedAlpha)
                self.layoutIfNeeded()
            }) { (_) in closure(self.customView) }
        case .top:
            shadowView.snp.remakeConstraints { (make) in
                make.centerX.equalToSuperview()
                make.top.equalToSuperview()
            }
            UIView.animate(withDuration: duration, animations: { [unowned self] in
                self.backgroundColor = self.coverColor.withAlphaComponent(self.backgounedAlpha)
                self.layoutIfNeeded()
            }) { (_) in closure(self.customView) }
        case .bottom:
            shadowView.snp.remakeConstraints { (make) in
                make.centerX.equalToSuperview()
                make.bottom.equalToSuperview()
            }
            UIView.animate(withDuration: duration, animations: { [unowned self] in
                self.backgroundColor = self.coverColor.withAlphaComponent(self.backgounedAlpha)
                self.layoutIfNeeded()
            }) { (_) in closure(self.customView) }
        }
    }
    
    private func dismissAnimate(completed closure: @escaping CompleteClosure) {
        /// 更新动画
        switch style {
        case .normal:
            UIView.animate(withDuration: duration, animations: { [unowned self] in
                self.shadowView.alpha = 0.0
                self.backgroundColor = self.coverColor.withAlphaComponent(0.0)
                self.layoutIfNeeded()
            }) { (_) in closure(self.customView) }
        case .top:
            shadowView.snp.remakeConstraints { (make) in
                make.centerX.equalToSuperview()
                make.bottom.equalTo(self.snp.top)
            }
            UIView.animate(withDuration: duration, animations: { [unowned self] in
                self.backgroundColor = self.coverColor.withAlphaComponent(0.0)
                self.layoutIfNeeded()
            }) { (_) in closure(self.customView) }
        case .bottom:
            shadowView.snp.remakeConstraints { (make) in
                make.centerX.equalToSuperview()
                make.top.equalTo(self.snp.bottom)
            }
            UIView.animate(withDuration: duration, animations: { [unowned self] in
                self.backgroundColor = self.coverColor.withAlphaComponent(0.0)
                self.layoutIfNeeded()
            }) { (_) in closure(self.customView) }
        }
    }
}
