//
//  CSVHelper.swift
//  CTFit
//
//  Created by Mac on 2020/12/14.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import Foundation

struct CSVFile {
    /// 文件 URL
    var fileUrl: URL
    /// 文件名
    var fileName: String { fileUrl.lastPathComponent }
    /// 最近修改时间
    var latestTimeStr: String {
        do {
            let lastDate = try fileUrl.resourceValues(forKeys: [.contentModificationDateKey]).contentModificationDate ?? Date.distantPast
            return CSVFile.dformatter.string(from: lastDate)
        } catch {
            return ""
        }
    }
    
    private static var dformatter: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "yyyy-MM-dd HH:mm:ss"
           return dformatter
    }()
}

class CSVHelper {
    
    typealias CompletedHandlerClosure = ()->Void
    
    enum CSVFileType {
        case user, total, step, heart, sleep
        var name: String {
            switch self {
            case .user:  return "User infomation.csv"
            case .total: return "Records for total.csv"
            case .step:  return "Records for step.csv"
            case .heart: return "Records for heart.csv"
            case .sleep: return "Records for sleep.csv"
            }
        }
    }
    
    static var directoryName: String = { "CSV" }()

    static var directoryURL: URL = {
        let documentURL = FileManager.default.urls(for: .documentDirectory, in: .userDomainMask)[0]
        let directoryURL = documentURL.appendingPathComponent(directoryName, isDirectory: true)
        return directoryURL
    } ()
    
    static func fileList() -> [CSVFile] {
        let fileManager = FileManager.default
        let properties: [URLResourceKey] = [.fileSizeKey, .contentModificationDateKey, .nameKey];
        let directoryOptions: FileManager.DirectoryEnumerationOptions = .skipsSubdirectoryDescendants;
        let urlItems = try? fileManager.contentsOfDirectory(at: directoryURL, includingPropertiesForKeys: properties, options: directoryOptions)
        if let hasUrlItems = urlItems {
            var fileItems = [CSVFile]()
            for url in hasUrlItems {
                if url.pathExtension == "csv" {
                    fileItems.append(CSVFile(fileUrl: url));
                }
            }
            return fileItems;
        }
        return [];
    }
    
    static func export(_ closure: @escaping CompletedHandlerClosure) {
        deleteAll()
        guard validCSVPath() else { closure(); return }
        DispatchQueue.global().async {
            onHandelCsvContent()
            DispatchQueue.main.async {
                closure()
            }
        }
    }
    
    @discardableResult
    static func deleteFile(_ url: URL) -> Bool {
        do {
            try FileManager.default.removeItem(at: url);
            return true
        } catch {
            return false
        }
    }
    
    @discardableResult
    static func deleteAll() -> Bool {
        deleteFile(directoryURL)
    }
}

extension CSVHelper {
    private static func validCSVPath() -> Bool {
        var directory: ObjCBool = ObjCBool(false)
        let exists = FileManager.default.fileExists(atPath: directoryURL.path, isDirectory: &directory)
        if exists && directory.boolValue{
        } else {
            do {
                try FileManager.default.createDirectory(atPath: directoryURL.path, withIntermediateDirectories: true, attributes: nil)
            } catch {
                return false
            }
        }
        return true
    }
    
    private static func fileUrl(for type: CSVFileType) -> URL {
        return directoryURL.appendingPathComponent(type.name, isDirectory: false)
    }
    
    private static func onWriteFile(for type: CSVFileType, _ content: String) {
        let url = fileUrl(for: type)
        do {
            try content.write(toFile: url.path, atomically: true, encoding: .utf8);
        } catch let error as NSError {
            print("Error occured when export \(type.name) file: " + error.description);
        }
    }
    
    private static func onHandelCsvContent() {
        switch BandConnect.protocolType {
        case .series_z: onHandlerUserInfo(); onHandlerRecords()
        case .series_s: onHandlerHybridInfo()
        }
    }
    
}

import BandKit
extension CSVHelper {
    private static func onHandlerUserInfo() {
        let content = SRUserInfo.csvContent(BandSettings.standard.user.nickname ?? "")
        onWriteFile(for: .user, content)
    }
    
    private static func onHandlerRecords() {
        var totalContent = SRTotal.csvTitle()
        var stepContent = SRStep.csvTitle()
        var heartContent = SRHeart.csvTitle()
        var sleepContent = SRSleep.csvTitle()
        BandRecord.csvQueryAll { (totalArray, stepArray, heartArray, sleepArray) in
            totalArray.forEach { (total) in totalContent += "\n" + total.csvText() }
            stepArray.forEach { (step) in  stepContent += "\n" + step.csvText() }
            heartArray.forEach { (heart) in  heartContent += "\n" + heart.csvText() }
            sleepArray.forEach { (sleep) in  sleepContent += "\n" + sleep.csvText() }
        }
        onWriteFile(for: .total, totalContent)
        onWriteFile(for: .step, stepContent)
        onWriteFile(for: .heart, heartContent)
        onWriteFile(for: .sleep, sleepContent)
    }
    
}

import YFitKit
extension CSVHelper {
    private static func onHandlerHybridInfo() {
        let content = YFHybridInfo.csvContent(BandSettings.standard.user.nickname ?? "")
        onWriteFile(for: .user, content)
    }
}
