//
//  LocationHelper.swift
//  CTFit
//
//  Created by Mac on 2020/6/12.
//  Copyright © 2020 shirajo. All rights reserved.
//


import UIKit
import CoreLocation

class LocationHelper: NSObject {
    
    typealias AuthorizationClosure = (CLAuthorizationStatus) -> Void
    typealias UpdateHeadingClosure = (CLHeading) -> Void
    typealias UpdateLocationClosure = (CLLocation) -> Void
    
    var didChangeAuthorizatio: AuthorizationClosure? = nil
    var didUpdateHeading: UpdateHeadingClosure? = nil
    var didUpdateLocation: UpdateLocationClosure? = nil
 
    static let shared = LocationHelper()
    
    // MARK: private

    private override init() {
        super.init()
        self.locationManager.delegate = self
    }
    
    private var locationManager: CLLocationManager = {
        let manager = CLLocationManager();
        manager.requestAlwaysAuthorization(); // 请求权限验证
        manager.desiredAccuracy = kCLLocationAccuracyBest; // 尽可能达到最佳的准确性
        manager.distanceFilter = 2; // 指定最小更新距离(单位为米)。
        manager.headingFilter = 1;  // 更新方向角度的最小偏差(箭头表示用户当前pos)
        manager.activityType = .fitness
        manager.pausesLocationUpdatesAutomatically = false // 指定可能时可自动暂停位置更新
        if #available(iOS 9.0, *) { // 开启后台定位
            manager.allowsBackgroundLocationUpdates = true;
        }
        return manager;
    }()
    
    private var hasPermissionForLocation: Bool {
        if LocationHelper.hasPermissionForLocation(),
            !(LocationHelper.runningInBackground() && LocationHelper.hasPermissionWhenInUseForLocation()) {
            return true
        }
        return false
    }
    
    // MARK: Public method
    func requestAuthorization() {
        locationManager.requestAlwaysAuthorization()
    }
    
    func requestLocation() {
        locationManager.requestLocation()
    }

    func startUpdatingLocation() {
        locationManager.startUpdatingLocation()
    }
    
    func stopUpdatingLocation() {
        locationManager.stopUpdatingLocation()
    }
    
    func startUpdatingHeading() {
        locationManager.startUpdatingHeading()
    }
    
    func stopUpdatingHeading() {
        locationManager.stopUpdatingHeading()
    }

}

// MARK: -  LatestLocation

extension LocationHelper {
    
    static let  kLatestLocationKey: String = "kLatestLocationKey"

    static var latestLocation: CLLocationCoordinate2D {
        get {
            if _latestLocation != nil { return _latestLocation! }
            
            if let cache = UserDefaults.standard.dictionary(forKey: kLatestLocationKey),
                let latitude: Double = cache["latitude"] as? Double,
                let longitude: Double = cache["longitude"] as? Double {
                _latestLocation = CLLocationCoordinate2D(latitude: latitude, longitude: longitude)
            } else {
                _latestLocation = CLLocationCoordinate2D(latitude:  41.03, longitude: 39.26)
            }
            return _latestLocation!
        } set {
            _latestLocation = newValue
        }
    }
    
    private static var _latestLocation: CLLocationCoordinate2D? {
        didSet {
            guard _latestLocation != oldValue else { return }
            let dictionary = ["latitude": _latestLocation?.latitude,"longitude": _latestLocation?.longitude]
            UserDefaults.standard.set(dictionary, forKey: kLatestLocationKey)
            UserDefaults.standard.synchronize()
        }
    }
}


// MARK: - CLLocationManagerDelegate

extension LocationHelper: CLLocationManagerDelegate {
    /// 改变授权状态
    func locationManager(_ manager: CLLocationManager, didChangeAuthorization status: CLAuthorizationStatus) {
        UILog.v("status: \(status.text)")
        switch status {
        case .denied, .notDetermined, .restricted: break
        case .authorizedAlways, .authorizedWhenInUse:
            LocationHelper.dismissNoLocationPermissionAlert()
        default:
            break
        }
    }
    
    /// 定位失败
    func locationManager(_ manager: CLLocationManager, didFailWithError error: Error) {
         UILog.v("error: \(error)")
    }
    
    /// 方向更新
    func locationManager(_ manager: CLLocationManager, didUpdateHeading newHeading: CLHeading) {
        // UILog.v("newHeading: \(newHeading)")
        guard let block = didUpdateHeading else { return }
        block(newHeading)
    }
    
    /// 用户位置更新
    func locationManager(_ manager: CLLocationManager, didUpdateLocations locations: [CLLocation]) {
        // UILog.v("locations: \(locations)")
        guard let block = didUpdateLocation, let newLocation = locations.first,
        newLocation.speed < 100 else { return }
        LocationHelper.latestLocation = newLocation.coordinate
        block(newLocation)
    }
}

extension CLAuthorizationStatus {
    var text: String {
        switch self {
        case .notDetermined: return "notDetermined" // 未选择
        case .restricted: return "restricted" // 授权限制
        case .denied: return "denied" // 否认
        case .authorizedAlways: return "authorizedAlways" // 总是允许
        case .authorizedWhenInUse: return "authorizedWhenInUse" // 使用时允许
        @unknown default: return "Error"
        }
    }
}


// MARK: - 定位权限

extension LocationHelper {
    
    private static func runningInBackground() -> Bool {
        return (UIApplication.shared.applicationState == .background);
    }
    
    // 判断当前定位是否可用，不可用弹窗提示
    private static func locationServicesEnabled () -> Bool {
        if !CLLocationManager.locationServicesEnabled() {
            print("设置打开定位授权")
            return false
        }
        return true
    }
    
    /// 同意了定位权限
    private static func hasPermissionForLocation() -> Bool {
        guard CLLocationManager.locationServicesEnabled() else { return false}
        let status = CLLocationManager.authorizationStatus()
        if status == .authorizedAlways || status == .authorizedWhenInUse {
            return true;
        }
        return false
    }
    
    /// Always 权限
    private static func hasPermissionAlwaysForLocation() -> Bool {
        guard CLLocationManager.locationServicesEnabled() else { return false}
        let status = CLLocationManager.authorizationStatus()
        if status == .authorizedAlways { return true; }
        return false
    }
    
    /// WhenInUse 权限
    private static func hasPermissionWhenInUseForLocation() -> Bool {
        guard CLLocationManager.locationServicesEnabled() else { return false}
        let status = CLLocationManager.authorizationStatus()
        if status == .authorizedWhenInUse { return true; }
        return false
    }
    
    
    /// 拒绝定位权限
    private static func hasDeniedUseForLocation() -> Bool {
        guard CLLocationManager.locationServicesEnabled() else { return true}
        let status = CLLocationManager.authorizationStatus()
        if status == .denied { return true; }
        return false
    }
    
    /// 用户还没有对这个应用程序作出选择
    private static func hasNoDeniedUseForLocation() -> Bool {
        guard CLLocationManager.locationServicesEnabled() else { return false}
        let status = CLLocationManager.authorizationStatus()
        if status == .notDetermined { return true; }
        return false
    }
    
    private static var noPermissionAlertVc: UIViewController? = nil
    
    
    
    private static func dismissNoLocationPermissionAlert() {
        guard hasPermissionForLocation() else { return}
        guard let hasAlert = noPermissionAlertVc else { return }
        hasAlert.dismiss(animated: true, completion: nil)
    }
    
    private static func openPermissionSetting() {
        guard let url = URL(string: UIApplication.openSettingsURLString) else { return }
        guard UIApplication.shared.canOpenURL(url) else { return }
        if #available(iOS 10.0, *) {
            UIApplication.shared.open(url)
        } else {
            UIApplication.shared.openURL(url)
        }
    }
    
    @discardableResult
    static func showNoLocationPermissionAlert() -> Bool {
        guard !hasPermissionForLocation() else { return false }
        guard noPermissionAlertVc == nil else { return false }
        let vc = AlertManager.show(title: nil, message: SRString.Sport.denied_permission_location_message.locastr, yesHander: { (_) in
            openPermissionSetting()
            noPermissionAlertVc = nil
        }) { (_) in  noPermissionAlertVc = nil }
        noPermissionAlertVc = vc
        return true
    }
}

// MARK: - 天气功能监听
import RxSwift
extension LocationHelper {

    private static var first: Bool = true
    private static var mainScheduler = MainScheduler.instance
    private static var disposeBag = DisposeBag()

    static func observeWeatherForBand() {
        BandData.updatedOutput.observeOn(mainScheduler).subscribe(onNext: { (dataType) in
            switch dataType {
            case .firmwareInfo:
                let feature = BandData.firmwareFeature
                if feature.weather, first {
                    _ = LocationHelper.shared
                    showNoLocationPermissionAlert()
                    first = false
                }
            default: break
            }
        }).disposed(by: disposeBag)
    }
}




