//
//  Language.swift
//  CTFit
//
//  Created by Mac on 2020/5/18.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit

private struct LanguageConstant {
    struct IdentifyKey {
        static let system: String       = "LanguageConstant.IdentifyKey.system";
        static let application: String  = "LanguageConstant.IdentifyKey.application";
    }
}

enum LanguageType: String {
    case en
    case zh_Hans
    case zh_Hant
    case ja
    case de
    case fr
    case ru
        
    case it
    case es
    case pt
    case hu
    case ro

    case sv
    case cs
    case da
    case el
    case nl
    case pl
    case th
    
    /// 对应语言文件标识
    var identify: String {
        switch self {
        case .en:       return "en"
        case .zh_Hans:  return "zh-Hans"
        case .zh_Hant:  return "zh-Hant"
        case .ja:       return "ja"
        case .de:       return "de"
        case .fr:       return "fr"
        case .ru:       return "ru"
                
        case .it:       return "it"
        case .es:       return "es"
        case .pt:       return "pt-PT"
        case .hu:       return "hu"
        case .ro:       return "ro"

        case .sv:       return "sv"
        case .cs:       return "cs"
        case .da:       return "da"
        case .el:       return "el"
        case .nl:       return "nl"
        case .pl:       return "pl"
        case .th:       return "th"
        }
    }
    
    /// 判断用的前缀
    var prefix: String {
        switch self {
        case .en:       return "en"
        case .zh_Hans:  return "zh-Hans"
        case .zh_Hant:  return "zh-Hant"
        case .ja:       return "ja"
        case .de:       return "de"
        case .fr:       return "fr"
        case .ru:       return "ru"
                
        case .it:       return "it"
        case .es:       return "es"
        case .pt:       return "pt"
        case .hu:       return "hu"
        case .ro:       return "ro"

        case .sv:       return "sv"
        case .cs:       return "cs"
        case .da:       return "da"
        case .el:       return "el"
        case .nl:       return "nl"
        case .pl:       return "pl"
        case .th:       return "th"
        }
    }
    
    /// 默认的语言类型
    static let defaultType: LanguageType = .en
    
    /// 支持的语言类型集合， 默认语言加入集合
    static let collection: [LanguageType] = [.en, .zh_Hans, .zh_Hans, .ja, .de, .fr, .ru,
                                             .it, .es, .pt, .hu, .ro,
                                             .sv, .cs, .da, .el, .nl, .pl, .th]
    
    /// 语言标识来解析出所支持的类型
    static func parse(_ identify: String) -> LanguageType{
        var type = defaultType
        for idx in 0..<collection.count {
            let item = collection[idx]
            if identify.hasPrefix(item.prefix) {
                type = item
                break
            }
        }
        if identify.hasPrefix("wuu-Hans") || identify.hasPrefix("yue-Hans") {
            return .zh_Hans
        }
        
        if identify.hasPrefix("zh-HK") || identify.hasPrefix("yue-Hant") {
            return .zh_Hant
        }
        return type
    }
}

extension LanguageType: CustomStringConvertible {
    var description: String {
        switch self {
        case .en:       return "英文";
        case .zh_Hans:  return "简体中文";
        case .zh_Hant:  return "繁体中文";
        case .ja:       return "日文";
        case .de:       return "德文";
        case .fr:       return "法文";
        case .ru:       return "俄文";
            
        case .it:       return "意大利文";
        case .es:       return "西班牙文";
        case .pt:       return "葡萄牙文";
        case .hu:       return "匈牙利文";
        case .ro:       return "罗马尼亚文";

        case .sv:       return "瑞典文";
        case .cs:       return "捷克文";
        case .da:       return "丹麦文";
        case .el:       return "希腊文";
        case .nl:       return "荷兰文";
        case .pl:       return "波兰文";
        case .th:       return "泰文";
        }
    }
}

class NELanguage {

    /// Base.lproj for bundle， 默认文件
    static let iproj_bundle_base: Bundle? = {
        if let path = Bundle.main.path(forResource: "en", ofType: "lproj") {
            return Bundle(path: path);
        }
        return nil;
    }()
    
    /// 当前系统语言标识
    static func latestSystemIdentity() -> String? {
        let identify = Locale.preferredLanguages.first
        return identify;
    }
    
    /// 上次的系统语言，在 UserDefaults 缓存中
    static func lastSystemIdentify() -> String? {
        let identify = UserDefaults.standard.string(forKey: LanguageConstant.IdentifyKey.system)
        return identify;
    }
    /// 保存系统语言标识
    static func saveSystem(_ identify: String) {
        UserDefaults.standard.set(identify, forKey: LanguageConstant.IdentifyKey.system)
        UserDefaults.standard.synchronize()
    }
    
    /// 上次的应用语言，在 UserDefaults 缓存中
    static func lastApplictionIdentify() -> String? {
        let identify = UserDefaults.standard.string(forKey: LanguageConstant.IdentifyKey.application)
        return identify;
    }
    /// 保存应用语言标识
    static func saveApplication(_ identify: String) {
        UserDefaults.standard.set(identify, forKey: LanguageConstant.IdentifyKey.application)
        UserDefaults.standard.synchronize()
    }
    
    /// 是否将 App 语言同步成系统语言,  第一次同步不在该判断内， 第一次直接同步
    static let systemSync: Bool = true;
    
    /// 应用语言文件的 Bundle
    private var iproj_bundle_appl: Bundle? = NELanguage.iproj_bundle_base
    
    /// 语言类型
    private var type: LanguageType = LanguageType.defaultType {
        didSet {
            guard type != oldValue else { return }
            updata_iproj_bundle_appl()
        }
    }
    
    private func updata_iproj_bundle_appl() {
        if let path = Bundle.main.path(forResource: type.identify, ofType: "lproj") {
            self.iproj_bundle_appl = Bundle(path: path);
        }
    }
    
    
    init() {
        
        do {
            UILog.i("main.preferredLocalizations: \(Bundle.main.preferredLocalizations)")
            // UILog.d("preferredLanguages: \(Locale.preferredLanguages)")
            // UILog.d("main.localizations: \(Bundle.main.localizations)")
        }

        do {
            if let identify = NELanguage.lastApplictionIdentify() {
                /// 应用语言
                type = LanguageType.parse(identify)
            } else {
                /// 首次， 从系统匹配
                if let latest = NELanguage.latestSystemIdentity() {
                    type = LanguageType.parse(latest)
                    NELanguage.saveApplication(latest)
                }
            }
        }

        do {
            /// 系统语言更改，去同步？
            if let latest = NELanguage.latestSystemIdentity(), let last = NELanguage.lastSystemIdentify(), latest != last, NELanguage.systemSync {
                type = LanguageType.parse(latest)
                NELanguage.saveApplication(latest)
            }
        }
        
        do {
            /// 缓存系统语言标识
            if let latest = NELanguage.latestSystemIdentity() {
                NELanguage.saveSystem(latest)
            }
        }
        
        updata_iproj_bundle_appl()
    }
    
    
    

    private func text(_ key: String, _ table: String? = nil) -> String {
        var value = "9527"
        if let bundle_appl = self.iproj_bundle_appl {
            value = bundle_appl.localizedString(forKey: key, value: "9527", table: table);
        }
        if let bundle_base = NELanguage.iproj_bundle_base, value == "9527" {
            value = bundle_base.localizedString(forKey: key, value: key, table: table);
        }
        if value == "9527" { value = key }
        return value;
    }
    

    // ------------------------------------------------------------------------
    //                            MARK: - Public 外部方法
    // ------------------------------------------------------------------------
    static let shared = NELanguage();
    
    static func type() -> LanguageType { return shared.type }

    static func setType(_ type: LanguageType) { shared.type = type }
    
    static func string(_ key: String) -> String { return shared.text(key) }

    static func stringTarget(_ key: String) -> String { return shared.text(key, "Target") }
    
    static func stringInfoPlist(_ key: String) -> String { return shared.text(key, "InfoPlist") }

}



public func NEString(_ key: String) -> String {
    return NELanguage.string(key);
}

public func NETargetString(_ key: String) -> String {
    return NELanguage.stringTarget(key);
}

@discardableResult
public func NEInfoPlistString(_ key: String) -> String {
    return NELanguage.stringInfoPlist(key);
}
