//
//  AutoConnectManager.swift
//  CTFit
//
//  Created by Mac on 2021/1/30.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import UIKit
import RxSwift

class AutoConnectManager {
    
    private var mainScheduler = MainScheduler.instance
    private var disposeBag = DisposeBag()
    private var peripheral: BandPeripheral?
    private var yfenable: Bool = false
    private let key = "kValidAutoConnectKey";
    private var valid: Bool {
        get {
            if let _ = UserDefaults.standard.value(forKey: key) {
                return UserDefaults.standard.bool(forKey: key)
            } else {
                return true
            }
        }
        set {
            UserDefaults.standard.setValue(newValue, forKey: key)
            UserDefaults.standard.synchronize()
        }
    }
}

extension AutoConnectManager {

    private func onAutoConnect() {
        observe()
        if BandConnect.centralState == .poweredOn { connectedToLocal() } 
    }
        
    private func observeBleState(){
        BandConnect.bleStateOutput.observeOn(MainScheduler.instance).subscribe(onNext: { [weak self] state in
            guard let strong = self else { return}
            switch state {
            case .poweredOn: strong.connectedToLocal()
            default:  break
            }
        }).disposed(by: disposeBag)
    }
    
    private func observe() {
        observeBleState()
        observeSearchdS()
        BandConnect.actionChangedOutput.observeOn(MainScheduler.instance).subscribe(onNext: {[weak self] state in
            guard let strong = self else { return }
            if case .success = state { strong.connectSuccess() }
        }).disposed(by: disposeBag)
        
        BandData.appNotifyOutput.subscribe(onNext: { [weak self] type in
            guard let strong = self else { return }
            switch type {
            case .didBecomeActive: strong.connect()
            default: break
            }
        }).disposed(by: disposeBag)
    }

    private func connect() {
        guard valid else { return }
        DispatchQueue.main.asyncAfter(deadline: .now()+0.1) { [weak self] in
            guard let strong = self, let band = strong.peripheral else { return }
            BandConnect.connect(for: band)
        }
    }
    
    private func connectSuccess() {
        guard valid else { return}
        valid = false
        BandConnect.bind()
    }
    
    private func connectedToLocal() {
        guard valid else { return }
        self.connectedToLocalZ()
        self.connectedToLocalS()
    }
}

extension AutoConnectManager {
    private func connectedToLocalZ() {
        guard valid else { return}
        let peripherals = BandSearch.retrieveConnectedPeripherals().filter({ $0.support && $0.protocolType == .series_z })
        if let p = peripherals.first { peripheral = p; connect() }
    }
}



import YFitKit
extension AutoConnectManager {
    
    private func connectedToLocalS() {
        guard valid else { return}
        guard peripheral == nil else { return}
        DispatchQueue.main.asyncAfter(deadline: .now()+0.1) {
            self.yfenable = true
            YFKitService.searchDeviceStart()
        }
        DispatchQueue.main.asyncAfter(deadline: .now()+0.3) {
            self.yfenable = false
            YFKitService.searchDeviceStop()
        }
    }
    
    private func observeSearchdS(){
        YFKitService.foundDeviceOutput.observeOn(mainScheduler).subscribe(onNext: { [weak self] peripherals in
            guard let strong = self, strong.yfenable else { return }
            if let p = peripherals.filter({ $0.rssiValue == 0 }).map({ BandPeripheral(searched: $0) }).filter({ $0.support }).first {
                strong.found(searched: p)
            }
        }).disposed(by: disposeBag)
    }
    
    private func found(searched peripheral: BandPeripheral) {
        guard valid, yfenable else { return}
        yfenable = false
        YFKitService.searchDeviceStop()
        self.peripheral = peripheral
        self.connect()
    }
}


extension AutoConnectManager {
    private static let shared = AutoConnectManager();
    
    static var valid: Bool {
        get { shared.valid }
        set { shared.valid = newValue }
    }
    
    static func autoConnect() {
        guard valid, !BandConnect.binded else { return }
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.2) {
            shared.onAutoConnect();
        }
    }
}
