//
//  PushNotifyManager.swift
//  CTFit
//
//  Created by Mac on 2020/6/1.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import Foundation
import UserNotifications

class PushNotifyManager {
    
    // 注册远程推送通知
    static func registerNotifications(_ application: UIApplication = UIApplication.shared) {
        /// Background Fetch
        application.setMinimumBackgroundFetchInterval(UIApplication.backgroundFetchIntervalMinimum);
        
        if #available(iOS 10.0, *) {
            let center = UNUserNotificationCenter.current()
            center.delegate = application as? UNUserNotificationCenterDelegate
            center.getNotificationSettings { (setting) in
                if setting.authorizationStatus == .notDetermined {
                    // 未注册
                    center.requestAuthorization(options: [.badge,.sound,.alert]) { (result, error) in
                        if(result){
                            if !(error != nil){
                                UILog.d("注册成功 获取dt")
                                DispatchQueue.main.async { application.registerForRemoteNotifications() }
                            }
                        } else{
                            UILog.d("用户不允许推送")
                        }
                    }
                } else if (setting.authorizationStatus == .denied){
                    // 申请用户权限被拒 -- 弹出页面提示用户去显示
                    UILog.d("申请用户权限被拒")
                } else if (setting.authorizationStatus == .authorized){
                    // 用户已授权（再次获取dt）
                    UILog.d("用户已授权")
                    DispatchQueue.main.async { application.registerForRemoteNotifications() }
                } else {
                    // 未知错误
                    UILog.d("未知错误")
                }
            }
        } else if #available(iOS 8.0, *) {
            let settings = UIUserNotificationSettings(types: [.alert, .badge, .sound], categories: nil);
            application.registerUserNotificationSettings(settings)
        } else {
            application.registerForRemoteNotifications()
        }
    }
    
    /// 检查权限状态
    static func checkPushNotification(checkNotificationStatus isEnable : ((Bool)->())? = nil){
        if #available(iOS 10.0, *) {
            UNUserNotificationCenter.current().getNotificationSettings(){ (setttings) in
                switch setttings.authorizationStatus{
                case .authorized:
                    UILog.d("已授权")
                    isEnable?(true)
                case .denied:
                    UILog.d("未授权")
                    isEnable?(false)
                case .notDetermined:
                    UILog.d("未注册")
                    isEnable?(false)
                case .provisional:
                    UILog.d("应用程序被授权发布非中断用户通知.")
                    isEnable?(false)
                case .ephemeral:
                    UILog.d("该应用程序暂时被授权发布通知, 仅适用于应用程序剪辑.")
                    isEnable?(false)
                default:
                    if setttings.authorizationStatus.rawValue == 4{
                        UILog.d("该应用程序暂时被授权发布通知, 仅适用于应用程序剪辑.")
                        isEnable?(false)
                    }
                    break
                }
            }
        } else {
            let isNotificationEnabled = UIApplication.shared.currentUserNotificationSettings?.types.contains(UIUserNotificationType.alert)
            if isNotificationEnabled == true{
                UILog.d("已授权")
                isEnable?(true)
            }else{
                UILog.d("未授权")
                isEnable?(false)
            }
        }
    }
    
    /// cancelAllLocalNotifications
    func cancelAllLocalNotifications() {
        UIApplication.shared.applicationIconBadgeNumber = 0;
        // TODO: 适配iOS10
        if #available(iOS 10, *) {
            UNUserNotificationCenter.current().removeAllDeliveredNotifications()
        } else {
            guard let hasLocalNotifications = UIApplication.shared.scheduledLocalNotifications else {
                return
            }
            for localNotification in hasLocalNotifications  {
                UIApplication.shared.cancelLocalNotification(localNotification)
            }
        }
    }
}

/** MARK: - 推送通知代理 */

extension AppDelegate {

    /** MARK: application notification delegate  **/
    
    func application(_ application: UIApplication, didRegister notificationSettings: UIUserNotificationSettings) {
        UILog.d("App push. notificationSettings: \(notificationSettings)");
        application.registerForRemoteNotifications();
    }
    
    func application(_ application: UIApplication, didRegisterForRemoteNotificationsWithDeviceToken deviceToken: Data) {
        let tokenString = deviceToken.hexadecima
        UILog.i("App push. deviceToken: \(tokenString)")
        submitToken(token: tokenString)
    }
    
    func application(_ application: UIApplication, didFailToRegisterForRemoteNotificationsWithError error: Error) {
        UILog.d("App push. registerForRemoteNotifications Error: \(error)");
    }
    
    func application(_ application: UIApplication, didReceiveRemoteNotification userInfo: [AnyHashable : Any], fetchCompletionHandler completionHandler: @escaping (UIBackgroundFetchResult) -> Void) {
        UILog.i("App push. RemotePush: \(userInfo)");
        self.performWakeUp();
        DispatchQueue.main.asyncAfter(deadline: .now() + 60) {
            completionHandler(.newData);
        }
    }
    
    /* MARK:  Background Fetch */
    func application(_ application: UIApplication, performFetchWithCompletionHandler completionHandler: @escaping (UIBackgroundFetchResult) -> Void) {
        UILog.i("Background fetch.");
        self.performWakeUp();
        DispatchQueue.main.asyncAfter(deadline: .now() + 60) {
            completionHandler(.newData);
        }
    }
}


import NEUtils
extension AppDelegate {
    private static let tokenKey = "deviceTokenKey";
    private static var latestToken: String? {
        get {
            return UserDefaults.standard.string(forKey: tokenKey)
        }
        set {
            UserDefaults.standard.set(newValue, forKey: tokenKey)
            UserDefaults.standard.synchronize()
        }
    }
    
    func performWakeUp() {
        DispatchQueue.main.async {
            BandData.performWakeUp()
        }
    }
    
    func submitToken(token: String) {
        guard token != AppDelegate.latestToken else { return }
        let bundle = AppUtils.shared.bundleId
        NENetworkingManger.apnsSubmit(bundle, UIDevice.udidString(), token) { response in
            switch response.result {
            case .success(_): AppDelegate.latestToken = token
            default: break;
            }
        }
    }
}
