//
//  CameraView.swift
//  CTFit
//
//  Created by Mac on 2020/11/3.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import AVFoundation
import Photos

class SRCameraView: UIView {
    
    ///: - Initialize
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    deinit {
        SRDeviceOrientationHelper.shared.stopDeviceOrientationNotifier()
    }
    
    init() {
        super.init(frame: .zero)
        SRDeviceOrientationHelper.shared.startDeviceOrientationNotifier { _ in }
        backgroundColor = Colors.black
        addSubviews()
        setupEvents()
        start()
    }
    
    ///: - View customer

    private func addSubviews() {
        let items = [previewViewContainer, buttonsContainer, spinner, permissionButton]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        previewViewContainer.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
        }
        
        buttonsContainer.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-UIDevice.bottomLayoutGuide)
            make.top.equalTo(previewViewContainer.snp.bottom)
            make.height.equalTo(90);
        }
        
        spinner.snp.makeConstraints { (make) in
            make.center.equalToSuperview()
        }
        
        permissionButton.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(8)
            make.right.equalToSuperview().offset(-8)
            make.top.equalToSuperview().offset(32)
        }
    }
    
    private lazy var focusView = UIView(frame: CGRect(x: 0, y: 0, width: 90, height: 90))
    
    private lazy var previewViewContainer: UIView = {
        let view = UIView()
        view.backgroundColor = Colors.black_less
        return view
    } ()
    
    private lazy var buttonsContainer: SRCameraBottomView = {
        let view = SRCameraBottomView()
        view.backgroundColor = Colors.black
        return view
    } ()
    
    private lazy var spinner:  UIActivityIndicatorView = {
        let view = UIActivityIndicatorView(style: .whiteLarge)
        view.color = Colors.gray
        return view
    } ()
    
    private lazy var permissionButton: UIButton = {
        let button = UIButton(style: Stylesheet.View.permissionButton)
        button.setTitle(SRString.Connected.denied_permission_camera_message.locastr, for: .normal)
        button.isHidden = true
        button.addTarget(self, action: #selector(toPermission), for: .touchUpInside)
        return button
    }()
    
    @objc
    private func toPermission() {
        Helper.openPermissionSetting()
    }
    
    var flashButton: UIButton { buttonsContainer.flashButton }
    
    // MARK: - Field
    typealias DidCapturePhoto = (UIImage) -> Void
    var didCapturePhoto: DidCapturePhoto?
    
    private lazy var albumsManager = SRAlbumsManager()
    private lazy var photoCapture = newPhotoCapture()
    private lazy var isInited = false
}

/**
 let semaphore = DispatchSemaphore(value: 1)
 semaphore.signal()
 semaphore.wait()
 */
extension SRCameraView {
    
    func onDidCapturePhoto(_ closure: @escaping DidCapturePhoto) {
        didCapturePhoto = closure
    }
    
    /// 启用相机
    private func start() {
        SRPermissionCheckable.Camera.doAfterPermissionCheck { [weak self] permission in
            guard let strongSelf = self else { return }
            strongSelf.permissionButton.isHidden = permission
            guard permission else { return }
            self?.photoCapture.start(with: strongSelf.previewViewContainer, completion: {
                DispatchQueue.main.async {
                    self?.isInited = true
                    self?.refreshFlashButton()
                }
            })
        }
    }
    
    /// 停用相机
    private func stopCamera() {
        photoCapture.stopCamera()
    }
    
    /// 拍照
    func shoot() {
        /// 防止连续点击多次造成奔溃
        guard buttonsContainer.shotButton.isEnabled else { return }
        buttonsContainer.shotButton.isEnabled = false
        photoCapture.shoot { [weak self] imageData in
            guard let strong = self else { return }
            guard let shotImage = UIImage(data: imageData) else { return }
            
            strong.photoCapture.stopCamera()
            var image = shotImage
            /// 如果从前置摄像头拍摄，翻转图像.
            if let device = strong.photoCapture.device,
               device.position == .front {
                image = strong.flipImage(image: image)
            }
            
            let noOrietationImage = image.resetOrientation()
            let resultImage = noOrietationImage.resizedImageIfNeeded()
            DispatchQueue.main.async {
                strong.didCapture(resultImage)
            }
        }
    }
    
    private func didCapture(_ image: UIImage) {
        setRecentImage(image)
        start()
        didCapturePhoto?(image)
        buttonsContainer.shotButton.isEnabled = true
    }
    
    /// 想强制相机输出为正方形图像
    private func cropImageToSquare(_ image: UIImage) -> UIImage {
        let orientation: UIDeviceOrientation = SRDeviceOrientationHelper.shared.currentDeviceOrientation
        var imageWidth = image.size.width
        var imageHeight = image.size.height
        switch orientation {
        /// 如果方向是横向的，交换宽度和高度
        case .landscapeLeft, .landscapeRight:
            imageWidth = image.size.height
            imageHeight = image.size.width
        default: break
        }
        
        // The center coordinate along Y axis
        let rcy = imageHeight * 0.5
        let rect = CGRect(x: rcy - imageWidth * 0.5, y: 0, width: imageWidth, height: imageWidth)
        let imageRef = image.cgImage?.cropping(to: rect)
        return UIImage(cgImage: imageRef!, scale: 1.0, orientation: image.imageOrientation)
    }
    
    /// 当图像从前置摄像头拍摄时使用。
    private func flipImage(image: UIImage!) -> UIImage! {
        let imageSize: CGSize = image.size
        UIGraphicsBeginImageContextWithOptions(imageSize, true, 1.0)
        let ctx = UIGraphicsGetCurrentContext()!
        ctx.rotate(by: CGFloat(Double.pi/2.0))
        ctx.translateBy(x: 0, y: -imageSize.width)
        ctx.scaleBy(x: imageSize.height/imageSize.width, y: imageSize.width/imageSize.height)
        ctx.draw(image.cgImage!, in: CGRect(x: 0.0,
                                            y: 0.0,
                                            width: imageSize.width,
                                            height: imageSize.height))
        let newImage: UIImage = UIGraphicsGetImageFromCurrentImageContext()!
        UIGraphicsEndImageContext()
        return newImage
    }
}

// MARK: - Show recent album photos

extension SRCameraView {
    func doAfterUpdateRecentImage() {
        SRPermissionCheckable.Library.doAfterPermissionCheck { [weak self] _ in
            guard let strong = self else { return }
            strong.showRecentAlbumImage()
        }
    }
    
    private func showRecentAlbumImage() {
        buttonsContainer.spinner.startAnimating()
        DispatchQueue.global(qos: .userInitiated).async { [weak self] in
            guard let strong = self else { return }
            strong.albumsManager.fetchAlbums()
            DispatchQueue.main.async {
                strong.buttonsContainer.spinner.stopAnimating()
                strong.setRecentImage(strong.albumsManager.defaultAlbum?.thumbnail)
            }
        }
    }
    
    func setRecentImage(_ image: UIImage?) {
        guard SRPhotoCaptureConfig.userRecentEnable else { return }
        buttonsContainer.latestImageButton.setImage(image, for: .normal)
    }
}


// MARK: - Setup events

extension SRCameraView: UIGestureRecognizerDelegate {
    
    private func setupEvents() {
        buttonsContainer.flashButton.isHidden = true
        buttonsContainer.flashButton.addTarget(self, action: #selector(flashButtonTapped), for: .touchUpInside)
        buttonsContainer.shotButton.addTarget(self, action: #selector(shotButtonTapped), for: .touchUpInside)
        buttonsContainer.flipButton.addTarget(self, action: #selector(flipButtonTapped), for: .touchUpInside)
        if SRPhotoCaptureConfig.userRecentEnable {
            buttonsContainer.latestImageButton.addTarget(self, action: #selector(latestImageTap), for: .touchUpInside)
        }
        
        // Focus
        let tapRecognizer = UITapGestureRecognizer(target: self, action: #selector(self.focusTapped(_:)))
        tapRecognizer.delegate = self
        previewViewContainer.addGestureRecognizer(tapRecognizer)
        
        // Zoom
        let pinchRecongizer = UIPinchGestureRecognizer(target: self, action: #selector(self.pinch(_:)))
        pinchRecongizer.delegate = self
        previewViewContainer.addGestureRecognizer(pinchRecongizer)
    }

    /// 聚焦
    @objc
    private func focusTapped(_ recognizer: UITapGestureRecognizer) {
        guard isInited else { return }
        SRPermissionCheckable.Camera.doAfterPermissionCheck { [weak self] _ in
            self?.focus(recognizer: recognizer)
        }
    }
    
    private func focus(recognizer: UITapGestureRecognizer) {
        let point = recognizer.location(in: previewViewContainer)
        
        // Focus the capture
        let viewsize = previewViewContainer.bounds.size
        let newPoint = CGPoint(x: point.x/viewsize.width, y: point.y/viewsize.height)
        photoCapture.focus(on: newPoint)
        
        // Animate focus view
        focusView.center = point
        configureFocusView(focusView)
        previewViewContainer.addSubview(focusView)
        animateFocusView(focusView)
    }
    
    /// 缩放变焦
    @objc
    private func pinch(_ recognizer: UIPinchGestureRecognizer) {
        guard isInited else { return }
        SRPermissionCheckable.Camera.doAfterPermissionCheck { [weak self] _ in
            self?.zoom(recognizer: recognizer)
        }
    }
    
    private func zoom(recognizer: UIPinchGestureRecognizer) {
        photoCapture.zoom(began: recognizer.state == .began, scale: recognizer.scale)
    }
    
    /// :  - 翻转切换摄像头
    @objc
    private func flipButtonTapped() {
        SRPermissionCheckable.Camera.doAfterPermissionCheck { [weak self] _ in
            self?.photoCapture.flipCamera {
                self?.refreshflipButton()
                self?.refreshFlashButton()
            }
        }
    }
    
    private func refreshflipButton() {
        if let device = self.photoCapture.device {
            let flashImage = (device.position == .front ? ImageRepo.Camera.lenses_front : ImageRepo.Camera.lenses_back)
            buttonsContainer.flipButton.setImage(flashImage, for: .normal)
        }
    }

    /// : - 点击拍照按钮
    @objc
    private func shotButtonTapped() {
        SRPermissionCheckable.Camera.doAfterPermissionCheck { [weak self] _ in
            self?.shoot()
        }
    }
    
    /// : - 闪关灯
    @objc
    private func flashButtonTapped() {
        photoCapture.tryToggleFlash()
        refreshFlashButton()
    }
    
    private func refreshFlashButton() {
        let flashImage = photoCapture.currentFlashMode.flashImage()
        buttonsContainer.flashButton.setImage(flashImage, for: .normal)
        buttonsContainer.flashButton.isHidden = !photoCapture.hasFlash
    }
    
    /// : - 相册
    @objc
    private func latestImageTap() {
        albumsManager.doAfterFetchAlbums {[weak self] (_) in
            guard let strong = self else { return }
            strong.toLibrary()
        }
    }
    
    private func toLibrary() {
        let vc = SRLibraryViewController(albumsManager: albumsManager, type: .device)
        push(vc)
    }
    
    private func push(_ vc: UIViewController) {
        guard let currentVc = Helper.currentVc as? SRCameraViewController else { return }
        currentVc.navigationController?.pushViewController(vc, animated: true)
    }
}

// MARK: - Bottome views

final class SRCameraBottomView: UIView {
    
    ///: - Initialize
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }
    
    private func addSubviews() {
        let items = [shotButton, leftButton, spinner, flipButton]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        shotButton.snp.makeConstraints { (make) in
            make.center.equalToSuperview()
            make.width.height.equalTo(90)
        }
        
        leftButton.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.normal);
            make.centerY.equalToSuperview()
            make.width.height.equalTo(self.leftButtonSize)
        }
        
        spinner.snp.makeConstraints { (make) in
            make.center.equalTo(leftButton.snp.center)
        }
        
        flipButton.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.right.equalToSuperview().offset(-Constraints.normal);
            make.width.height.equalTo(40)
        }
    }
    
    ///: - View customer
    
    private var leftButton: UIButton { SRPhotoCaptureConfig.userRecentEnable ? latestImageButton: flashButton }

    private var leftButtonSize: CGFloat { SRPhotoCaptureConfig.userRecentEnable ? 45: 40 }
    
    fileprivate  lazy var flipButton: UIButton = {
        let button = UIButton()
        button.contentMode = .scaleAspectFill
        button.setImage(ImageRepo.Camera.lenses_back, for: .normal)
        return button
    } ()
    
    fileprivate  lazy var latestImageButton: UIButton = {
        let button = UIButton()
        button.backgroundColor = Colors.darkGray
        button.imageView?.contentMode = .scaleAspectFill
        button.layer.cornerRadius = 2
        return button
    } ()
    
    fileprivate  lazy var spinner:  UIActivityIndicatorView = {
        let view = UIActivityIndicatorView(style: .gray)
        return view
    } ()
    
    fileprivate  lazy var shotButton: UIButton = {
        let button = UIButton()
        button.contentMode = .scaleAspectFill
        let image = ImageRepo.Camera.tack_photo
        button.setImage(image, for: .normal)
        return button
    } ()
    
    lazy var flashButton: UIButton = {
        let button = UIButton()
        button.contentMode = .scaleAspectFill
        button.heightAnchor.constraint(equalToConstant: 30).isActive = true;
        button.widthAnchor.constraint(equalToConstant: 30).isActive = true;
        return button
    } ()
}


