//
//  SRCameraViewController.swift
//  CTFit
//
//  Created by Mac on 2020/11/3.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit

enum SRCameraType {
    case device, portrait, dialgroud
    
    var text: String {
        SRString.Connected.take_picture.locastr
    }
    
    var libraryType: SRLibraryType {
        switch self {
        case .device: return .device
        case .portrait: return .portrait
        case .dialgroud: return .dialgroud
        }
    }
}

class SRCameraViewController: BaseViewController, CustomView {
    
    typealias ViewClass = SRCameraView
    
    private override init() {
        super.init()
    }
    
    convenience init(type: SRCameraType = .portrait) {
        self.init()
        self.type = type
        navigationItem.title = type.text
        loadCaptureStyle()
    }
    
    override func loadView() {
        super.loadView()
        view = ViewClass()
    }
    
    override func initNavigationBar() {
        super.initNavigationBar()
        navigationItem.leftBarButtonItem = cancelBarItem()
        if type == .device {
            navigationItem.rightBarButtonItem = UIBarButtonItem(customView: customView.flashButton)
        }
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        view.backgroundColor = Colors.black
        swipe()
        onDidCapturePhoto()
        observe()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        if SRPhotoCaptureConfig.userRecentEnable { updateRecentImage()  }
        deviceEnter()
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        deviceExit()
    }
    

    private func swipe() {
        let swipeRecognizer = UISwipeGestureRecognizer(target: self, action: #selector(onCancel))
        swipeRecognizer.direction = .right
        view.addGestureRecognizer(swipeRecognizer)
    }
    
    ///:  - Field
    typealias DidCapturePhoto = (UIImage) -> Void
    private var type: SRCameraType = .portrait
    private var didCapturePhoto: DidCapturePhoto?
}

// MARK: - Method for customer

extension SRCameraViewController {
    
    public func onDidCapturePhoto(_ closure: @escaping DidCapturePhoto) {
        didCapturePhoto = closure
    }
    
    private func onDidCapturePhoto() {
        customView.onDidCapturePhoto { [weak self] (image) in
            guard let strong = self else { return }
            strong.capturePhoto(image)
        }
    }
    
    private func capturePhoto(_ image: UIImage) {
        SRPhotoSaver.albumTrySaveImage(image)
        if let cropVc = SRCropViewController.enter(image, type: type.libraryType) {
            cropVc.onDidFinishCropping { [weak self] (image) in
                guard let strong = self else { return }
                strong.didCapturePhoto?(image)
            }
        } else {
            didCapturePhoto?(image)
        }
    }
    
    private func loadCaptureStyle() {
        switch type {
        case .device:
            SRCropConfig.showsCrop = .none
            SRPhotoCaptureConfig.userRecentEnable = true
            SRPhotoCaptureConfig.shouldSaveNewPicturesToAlbum = true
        case .portrait, .dialgroud:
            SRCropConfig.showsCrop = .rectangle(ratio: 1.0)
            SRPhotoCaptureConfig.userRecentEnable = false
            SRPhotoCaptureConfig.shouldSaveNewPicturesToAlbum = true
        }
    }
    
    private func updateRecentImage() {
        customView.doAfterUpdateRecentImage()
    }
}

// MARK: - Observer
extension SRCameraViewController {
    
    private func observe() {
        guard type == .device else { return }
        observeDeviceNotify()
        observeAppNotify()
    }
    
    private func observeDeviceNotify() {
        
        BandConnect.statusChangedOutput.observeOn(mainScheduler).subscribe(onNext: { [weak self] status in
            guard let strong = self, case .connected = status else { return }
            strong.deviceEnter()
        }).disposed(by: disposeBag)
        
        BandData.updatedOutput.observeOn(mainScheduler).subscribe(onNext: { [weak self] (dataType) in
            guard let strong = self else { return }
            guard case BDDataType.notify(let notifyType) = dataType,
                  case BDNotifyType.camera(let cameraType) = notifyType,
                  cameraType == .takePicture else { return }
            strong.customView.shoot()
        }).disposed(by: disposeBag)
    }
    
    private func observeAppNotify() {
        BandData.appNotifyOutput.subscribe(onNext: { [weak self] type in
            guard let strong = self else { return }
            guard strong.active else { return}
            switch type {
            case .willEnterForeground:
                strong.deviceEnter()
                strong.updateRecentImage()
            case .didEnterBackground, .willTerminate:
                strong.deviceExit()
            default: break
            }
        }).disposed(by: disposeBag)
    }
    
    private func deviceEnter() {
        guard type == .device else { return }
        BandCommand.camera(for: .on)
    }
    
    private func deviceExit() {
        guard type == .device else { return }
        BandCommand.camera(for: .off)
    }
}

