//
//  SRCaptureDevicePermissionCheckable.swift
//  CTFit
//
//  Created by Mac on 2020/11/3.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import Photos
import AVFoundation

class SRPermissionCheckable {
    
    // MARK: - Library Permissions
    class Library {
        
        static func checkPermission() {
            checkPermissionToAccessPhotoLibrary { (_) in }
        }
        
        static func doAfterPermissionCheck(block:@escaping (Bool) -> Void) {
            checkPermissionToAccessPhotoLibrary { hasPermission in
                block(hasPermission)
            }
        }
            
        static var authorizationStatus: PHAuthorizationStatus? = nil
        
        static var isAuthorized: Bool {
            if authorizationStatus == .authorized { return true}
            if #available(iOS 14, *),  authorizationStatus == .limited { return true }
            return false
        }
        
        static var isLimited: Bool {
            if #available(iOS 14, *) { return (authorizationStatus == .limited) }
            return false
        }

        /// 如果.notDetermined, Async because将提示permission
        /// 并询问自定义弹出框是否被拒绝。
        private static func checkPermissionToAccessPhotoLibrary(block: @escaping (Bool) -> Void) {
            if #available(iOS 14, *) {
                let level: PHAccessLevel = .readWrite
                let status = PHPhotoLibrary.authorizationStatus(for: level)
                authorizationStatus = status
                switch status {
                case .authorized: block(true)
                case .limited: block(true)
                case .restricted, .denied: popup { block(false) }
                case .notDetermined:
                    PHPhotoLibrary.requestAuthorization(for: level) { s in
                        authorizationStatus = s
                        DispatchQueue.main.async {
                            block((s == .authorized || s == .limited))
                        }
                    }
                @unknown default: fatalError()
                }
            } else {
                let status = PHPhotoLibrary.authorizationStatus()
                authorizationStatus = status
                switch status {
                case .authorized: block(true)
                case .limited: block(true)
                case .restricted, .denied: popup { block(false) }
                case .notDetermined:
                    PHPhotoLibrary.requestAuthorization { s in
                        authorizationStatus = status
                        DispatchQueue.main.async {
                            block((s == .authorized))
                        }
                    }
                @unknown default: fatalError()
                }
            }
        }
        
        private static func popup(_ finishHandler: @escaping () -> Void)  {
            finishHandler()
            AlertManager.show(title: nil, message: SRString.Connected.denied_permission_library_message.locastr, okHander: { (_) in
                Helper.openPermissionSetting()
            }, cancelHander: { (_) in
                Helper.currentVc?.navigationController?.dismiss(animated: true)
            })
        }
        
    }
    
    // MARK: - Camera Permissions
    class Camera {
        
        static func checkPermission() {
            checkPermissionToAccessVideo { _ in }
        }
        
        static func doAfterPermissionCheck(block:@escaping (Bool) -> Void) {
            checkPermissionToAccessVideo { hasPermission in
                block(hasPermission)
            }
        }
        
        static var granted: Bool = false

        /// 如果.notDetermined, Async because将提示permission
        /// 并询问自定义弹出框是否被拒绝。
        private static func checkPermissionToAccessVideo(block: @escaping (Bool) -> Void) {
            let status = AVCaptureDevice.authorizationStatus(for: AVMediaType.video)
            granted = (status == .authorized)
            switch status {
            case .authorized:  block(true)
            case .restricted, .denied: popup { block(false) }
            case .notDetermined:
                AVCaptureDevice.requestAccess(for: .video, completionHandler: { _granted in
                    granted = _granted
                    DispatchQueue.main.async {
                        block(_granted)
                    }
                })
            @unknown default:
                fatalError()
            }
        }
        
        private static func popup(_ finishHandler: @escaping () -> Void)  {
            finishHandler()
            AlertManager.show(title: nil, message: SRString.Connected.denied_permission_camera_message.locastr, okHander: { (_) in
                Helper.openPermissionSetting()
            }, cancelHander: { (_) in
                Helper.currentVc?.navigationController?.dismiss(animated: true)
            })
        }
    }
}


