//
//  SRPhotoCaptureConfig.swift
//  CTFit
//
//  Created by Mac on 2020/11/3.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import Foundation
import AVFoundation

public struct SRPhotoCaptureConfig {
    
    /// 启用默认选择后置摄像头
    public static var devicePosition: AVCaptureDevice.Position = .back
    
    /// 定义相机的最大变焦因子。禁用相机变焦1。默认是1
    public static var maxCameraZoomFactor: CGFloat = 3
    
    /// 定义在用户的图片库中保存图片时相册的名称.
    public static var albumName: String = AppUtils.shared.name
    
    /// 是否保存图片到相册，默认为 true
    public static var shouldSaveNewPicturesToAlbum = true

    /// 例如: cappedTo: 1024将确保图片库或相机将调整大小，以适应一个1024x1024的盒子。
    /// 默认为原始图像大小。
    public static var targetImageSize = SRImageSize.original
    
    /// 用户拍下的最近照片可用性，默认：false
    public static var userRecentEnable = false
    
}


public enum SRImageSize {
    case original
    case cappedTo(size: CGFloat)
}

extension UIImage {
    
    func resized(to size: CGSize) -> UIImage? {
        UIGraphicsBeginImageContextWithOptions(size, false, scale)
        defer { UIGraphicsEndImageContext() }
        draw(in: CGRect(origin: .zero, size: size))
        return UIGraphicsGetImageFromCurrentImageContext()
    }
    
    func resetOrientation() -> UIImage {
        // Image has no orientation, so keep the same
        if imageOrientation == .up {
            return self
        }
        
        // Process the transform corresponding to the current orientation
        var transform = CGAffineTransform.identity
        switch imageOrientation {
        case .down, .downMirrored:           // EXIF = 3, 4
            transform = transform.translatedBy(x: size.width, y: size.height)
            transform = transform.rotated(by: CGFloat(Double.pi))
            
        case .left, .leftMirrored:           // EXIF = 6, 5
            transform = transform.translatedBy(x: size.width, y: 0)
            transform = transform.rotated(by: CGFloat(Double.pi / 2))
            
        case .right, .rightMirrored:          // EXIF = 8, 7
            transform = transform.translatedBy(x: 0, y: size.height)
            transform = transform.rotated(by: -CGFloat((Double.pi / 2)))
        default:
            ()
        }
        
        switch imageOrientation {
        case .upMirrored, .downMirrored:     // EXIF = 2, 4
            transform = transform.translatedBy(x: size.width, y: 0)
            transform = transform.scaledBy(x: -1, y: 1)
            
        case .leftMirrored, .rightMirrored:   //EXIF = 5, 7
            transform = transform.translatedBy(x: size.height, y: 0)
            transform = transform.scaledBy(x: -1, y: 1)
        default:
            ()
        }
        
        // Draw a new image with the calculated transform
        let context = CGContext(data: nil,
                                width: Int(size.width),
                                height: Int(size.height),
                                bitsPerComponent: cgImage!.bitsPerComponent,
                                bytesPerRow: 0,
                                space: cgImage!.colorSpace!,
                                bitmapInfo: cgImage!.bitmapInfo.rawValue)
        context?.concatenate(transform)
        switch imageOrientation {
        case .left, .leftMirrored, .right, .rightMirrored:
            context?.draw(cgImage!, in: CGRect(x: 0, y: 0, width: size.height, height: size.width))
        default:
            context?.draw(cgImage!, in: CGRect(x: 0, y: 0, width: size.width, height: size.height))
        }
        
        if let newImageRef =  context?.makeImage() {
            let newImage = UIImage(cgImage: newImageRef)
            return newImage
        }
        
        // In case things go wrong, still return self.
        return self
    }
    
    /// 如果需要进一步缩小图像大小，目标尺寸是上限。
    func resizedImageIfNeeded() -> UIImage {
        if case let SRImageSize.cappedTo(size: capped) = SRPhotoCaptureConfig.targetImageSize {
            let size = cappedSize(for: self.size, cappedAt: capped)
            if let resizedImage = self.resized(to: size) {
                return resizedImage
            }
        }
        return self
    }
    
    private  func cappedSize(for size: CGSize, cappedAt: CGFloat) -> CGSize {
        var cappedWidth: CGFloat = 0
        var cappedHeight: CGFloat = 0
        if size.width > size.height {
            // Landscape
            let heightRatio = size.height / size.width
            cappedWidth = min(size.width, cappedAt)
            cappedHeight = cappedWidth * heightRatio
        } else if size.height > size.width {
            // Portrait
            let widthRatio = size.width / size.height
            cappedHeight = min(size.height, cappedAt)
            cappedWidth = cappedHeight * widthRatio
        } else {
            // Squared
            cappedWidth = min(size.width, cappedAt)
            cappedHeight = min(size.height, cappedAt)
        }
        return CGSize(width: cappedWidth, height: cappedHeight)
    }
    
    func toCIImage() -> CIImage? {
        return self.ciImage ?? CIImage(cgImage: self.cgImage!)
    }
}
