//
//  SRPhotoCaptureHelper.swift
//  CTFit
//
//  Created by Mac on 2020/11/3.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import Foundation
import AVFoundation

enum SRFlashMode {
    case off
    case on
    case auto
}

extension SRFlashMode {
    func flashImage() -> UIImage {
        switch self {
        case .on: return ImageRepo.Camera.flashlight_on
        case .off: return ImageRepo.Camera.flashlight_off
        case .auto: return ImageRepo.Camera.flashlight_auto
        }
    }
}

// MARK: - Global functions
func newPhotoCapture() -> SRPhotoCapture {
    if #available(iOS 10.0, *) {
        return SRPhotoCaptureIOS10After()
    } else {
        return SRPhotoCaptureIOS10Before()
    }
}

func deviceForPosition(_ p: AVCaptureDevice.Position) -> AVCaptureDevice? {
    for device in AVCaptureDevice.devices(for: AVMediaType.video) where device.position == p {
        return device
    }
    return nil
}

func setFocusPointOnDevice(device: AVCaptureDevice, point: CGPoint) {
    do {
        try device.lockForConfiguration()
        if device.isFocusModeSupported(AVCaptureDevice.FocusMode.autoFocus) {
            device.focusMode = AVCaptureDevice.FocusMode.autoFocus
            device.focusPointOfInterest = point
        }
        if device.isExposureModeSupported(AVCaptureDevice.ExposureMode.continuousAutoExposure) {
            device.exposureMode = AVCaptureDevice.ExposureMode.continuousAutoExposure
            device.exposurePointOfInterest = point
        }
        device.unlockForConfiguration()
    } catch _ {
        return
    }
}

func toggledPositionForDevice(_ device: AVCaptureDevice) -> AVCaptureDevice.Position {
    return (device.position == .front) ? .back : .front
}

func flippedDeviceInputForInput(_ input: AVCaptureDeviceInput) -> AVCaptureDeviceInput? {
    let p = toggledPositionForDevice(input.device)
    let aDevice = deviceForPosition(p)
    return try? AVCaptureDeviceInput(device: aDevice!)
}

func configureFocusView(_ v: UIView) {
    v.alpha = 0.0
    v.backgroundColor = UIColor.clear
    v.layer.borderColor = UIColor(red: 1, green: 0.83, blue: 0, alpha: 0.95).cgColor
    v.layer.borderWidth = 1.0
    v.transform = CGAffineTransform(scaleX: 1.0, y: 1.0)
}

func animateFocusView(_ v: UIView) {
    UIView.animate(withDuration: 0.8, delay: 0.0, usingSpringWithDamping: 0.8,
                   initialSpringVelocity: 3.0, options: UIView.AnimationOptions.curveEaseIn,
                   animations: {
                    v.alpha = 1.0
                    v.transform = CGAffineTransform(scaleX: 0.7, y: 0.7)
    }, completion: { _ in
        v.transform = CGAffineTransform(scaleX: 1.0, y: 1.0)
        v.removeFromSuperview()
    })
}


extension AVCaptureSession {
    func resetInputs() {
        /// 移除所有sesison输入
        for i in inputs { removeInput(i) }
    }
}
