//
//  SRPhotoSaver.swift
//  CTFit
//
//  Created by Mac on 2020/11/4.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import Photos

public class SRPhotoSaver {
    
    class func albumTrySaveImage(_ image: UIImage) {
        guard SRPhotoCaptureConfig.shouldSaveNewPicturesToAlbum else { return }
        guard SRPermissionCheckable.Library.isAuthorized else { return }
        if SRPermissionCheckable.Library.isLimited {
            saveImage(image)
        } else {
            trySaveImage(image, inAlbumNamed: SRPhotoCaptureConfig.albumName)
        }
    }
    
    private  class func trySaveImage(_ image: UIImage, inAlbumNamed: String) {
        if SRPermissionCheckable.Library.isAuthorized {
            if let album = album(named: inAlbumNamed) {
                saveImage(image, toAlbum: album)
            } else {
                createAlbum(withName: inAlbumNamed) {
                    if let album = album(named: inAlbumNamed) {
                        saveImage(image, toAlbum: album)
                    }
                }
            }
        }
    }
    
    private class func saveImage(_ image: UIImage) {
        UIImageWriteToSavedPhotosAlbum(image, self, nil, nil)
    }
    
    private class func saveImage(_ image: UIImage, toAlbum album: PHAssetCollection) {
        PHPhotoLibrary.shared().performChanges({
            let changeRequest = PHAssetChangeRequest.creationRequestForAsset(from: image)
            let albumChangeRequest = PHAssetCollectionChangeRequest(for: album)
            let enumeration: NSArray = [changeRequest.placeholderForCreatedAsset!]
            albumChangeRequest?.addAssets(enumeration)
        }, completionHandler: { success, err in
            UILog.v("success: \(success), err: \(String(describing: err))")
        })
    }
    
    private class func createAlbum(withName name: String, completion:@escaping () -> Void) {
        PHPhotoLibrary.shared().performChanges({
            PHAssetCollectionChangeRequest.creationRequestForAssetCollection(withTitle: name)
        }, completionHandler: { success, err in
            UILog.v("success: \(success), err: \(String(describing: err))")
            if success {
                completion()
            }
        })
    }
    
    private  class func album(named: String) -> PHAssetCollection? {
        let fetchOptions = PHFetchOptions()
        fetchOptions.predicate = NSPredicate(format: "title = %@", named)
        let collection = PHAssetCollection.fetchAssetCollections(with: .album,
                                                                 subtype: .any,
                                                                 options: fetchOptions)
        return collection.firstObject
    }
}
