//
//  SRCropView.swift
//  CTFit
//
//  Created by Mac on 2020/11/9.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import SnapKit

enum SRCropDirectionType {
    case left, top, right, bottom
}

class SRCropView: UIView {    

    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    required init(image: UIImage, ratio: Double, type: SRLibraryType = .portrait) {
        self.image = image
        self.ratio = ratio
        self.type = type
        super.init(frame: .zero)
        clipsToBounds = true
        backgroundColor = groundColor
        setupSubviews()
        self.cropAreaView.setImage(image)
    }
    
    ///: - View customer
    private var ratio: Double
    private var image: UIImage
    private lazy var groundColor: UIColor = Colors.black
    
    private lazy var bottomView: SRCropBottomView = {
        let view = SRCropBottomView(type: type)
        view.backgroundColor = Colors.black
        view.onTap { self.cancel() } done: { self.done() }
        return view
    } ()
    
    private lazy var contentView: UIView = {
        let view = UIView()
        return view
    } ()
    
    private lazy var cropAreaView: SRCropAreaView = {
        let view = SRCropAreaView()
        return view
    } ()
    
    private func setupSubviews() {
        let items = [contentView, bottomView]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        contentView.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
        }
        bottomView.snp.makeConstraints { (make) in
            make.left.right.bottom.equalToSuperview()
            make.top.equalTo(contentView.snp.bottom)
        }
        setupContentViews()
    }
    
    private func setupContentViews() {
        let items = [cropAreaView]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            contentView.addSubview(view)
        }
        cropAreaView.snp.makeConstraints { (make) in
            make.center.equalToSuperview()
            make.width.equalToSuperview().offset(-Constraints.normal * 2)
            make.width.equalTo(cropAreaView.snp.height).multipliedBy(ratio)
        }
        
        loadCurtainView(.top)
        loadCurtainView(.right)
        loadCurtainView(.bottom)
        loadCurtainView(.left)
        loadBorders(.top)
        loadBorders(.right)
        loadBorders(.bottom)
        loadBorders(.left)
    }
    
    private func loadCurtainView(_ directionType: SRCropDirectionType) {
        let view = UIView()
        view.backgroundColor = groundColor.withAlphaComponent(0.7)
        view.translatesAutoresizingMaskIntoConstraints = false
        contentView.addSubview(view)
        switch directionType {
        case .top:
            view.snp.makeConstraints { (make) in
                make.left.right.top.equalToSuperview()
                make.bottom.equalTo(cropAreaView.snp.top)
            }
        case .bottom:
            view.snp.makeConstraints { (make) in
                make.left.right.bottom.equalToSuperview()
                make.top.equalTo(cropAreaView.snp.bottom)
            }
            
        case .left:
            view.snp.makeConstraints { (make) in
                make.left.equalToSuperview()
                make.top.equalTo(cropAreaView.snp.top)
                make.bottom.equalTo(cropAreaView.snp.bottom)
                make.right.equalTo(cropAreaView.snp.left)
            }
        case .right:
            view.snp.makeConstraints { (make) in
                make.right.equalToSuperview()
                make.top.equalTo(cropAreaView.snp.top)
                make.bottom.equalTo(cropAreaView.snp.bottom)
                make.left.equalTo(cropAreaView.snp.right)
            }
        }
    }
    
    private func loadBorders(_ directionType: SRCropDirectionType) {
        let line = separatorLine()
        cropAreaView.addSubview(line)
        let coarse = 0.5
        switch directionType {
        case .top:
            line.snp.makeConstraints { (make) in
                make.left.right.top.equalToSuperview()
                make.height.equalTo(coarse)
            }
        case .bottom:
            line.snp.makeConstraints { (make) in
                make.left.right.bottom.equalToSuperview()
                make.height.equalTo(coarse)
            }
            
        case .left:
            line.snp.makeConstraints { (make) in
                make.left.top.bottom.equalToSuperview()
                make.width.equalTo(coarse)
            }
        case .right:
            line.snp.makeConstraints { (make) in
                make.right.top.bottom.equalToSuperview()
                make.width.equalTo(coarse)
            }
        }
    }
    
    private func separatorLine() -> UIView {
        let view = UIImageView()
        let color = UIColor.white
        view.backgroundColor = color
        view.layer.shadowColor = UIColor.black.cgColor
        view.layer.shadowOpacity = 1
        view.layer.shadowRadius = 2
        view.layer.shadowOffset = CGSize(width: 0, height: 0)
        return view
    }
    
    // MARK: - Field
    typealias DidFinishCropping = (_ image: UIImage) -> Void
    private var didFinishCropping: DidFinishCropping?
    private let pinchGR = UIPinchGestureRecognizer()
    private let panGR = UIPanGestureRecognizer()

    private var type: SRLibraryType

}

extension SRCropView {
    func onDidFinishCropping(_ closure: @escaping DidFinishCropping) {
        didFinishCropping = closure
    }
    
    private func cancel() {
        guard let vc = Helper.currentVc as? SRCropViewController else { return }
        vc.navigationController?.dismiss(animated: true, completion: nil)
    }
    
    private func done() {
        guard let croppedImage = cropAreaView.cropImage() else { return }
        didFinishCropping?(croppedImage)
    }
}

class SRCropBottomView: UIView {
    typealias TapClosure = ()->Void
    private var cancelClosure: TapClosure?
    private var doneClosure: TapClosure?
    private var type: SRLibraryType

    func onTap(cancel: @escaping TapClosure, done: @escaping TapClosure) {
        self.cancelClosure = cancel
        self.doneClosure = done
    }
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    required init(type: SRLibraryType = .portrait) {
        self.type = type
        super.init(frame: .zero)
        setupSubviews()
    }
    
    ///: - View customer
    
    private lazy var toolbar: UIToolbar = {
        let bar = UIToolbar()
        bar.backgroundColor = Colors.black
        bar.setBackgroundImage(UIImage(), forToolbarPosition: .any, barMetrics: .default)
        bar.setShadowImage(UIImage(), forToolbarPosition: .any)
        return bar
    } ()

    private func setupSubviews() {
        let items = [toolbar]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        toolbar.snp.makeConstraints { (make) in
            make.top.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-UIDevice.bottomLayoutGuide)
        }
        setupToolbar()
    }
    
    private func setupToolbar() {
        let color = Colors.white
        let cancelButton = UIBarButtonItem(title: SRString.Base.cancel.locastr, style: .plain, target: self, action: #selector(cancel))
        cancelButton.tintColor = color
        let flexibleSpace = UIBarButtonItem(barButtonSystemItem: .flexibleSpace, target: nil, action: nil)
        let saveText = (type == .dialgroud) ? SRString.Base.ok.locastr : SRString.Base.save.locastr
        let saveButton = UIBarButtonItem(title: saveText, style: .plain, target: self, action: #selector(done))
        saveButton.tintColor = color
        toolbar.items = [cancelButton, flexibleSpace, saveButton]
    }

    @objc private func cancel() { cancelClosure?() }
    
    @objc private func done() { doneClosure?() }
}
