//
//  SRCropZoomableView.swift
//  CTFit
//
//  Created by Mac on 2020/11/17.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import UIKit
import Photos

protocol SRCropZoomableViewDelegate: AnyObject {
    func srAssetZoomableViewDidLayoutSubviews(_ zoomableView: SRCropZoomableView)
    func srAssetZoomableViewScrollViewDidZoom()
    func srAssetZoomableViewScrollViewDidEndZooming()
}

final class SRCropZoomableView: UIScrollView {
    public weak var myDelegate: SRCropZoomableViewDelegate?
    private var cropAreaDidChange = {}
    public var photoImageView = UIImageView()
    
    // 可以是视频预览图像查看或照片图像查看
    public var assetImageView: UIImageView {
        return photoImageView
    }
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        backgroundColor = Colors.black
        frame.size = CGSize.zero
        clipsToBounds = false
        photoImageView.frame = CGRect(origin: CGPoint.zero, size: CGSize.zero)
        maximumZoomScale = 6.0
        minimumZoomScale = 1
        showsHorizontalScrollIndicator = false
        showsVerticalScrollIndicator = false
        delegate = self
        alwaysBounceHorizontal = true
        alwaysBounceVertical = true
        isScrollEnabled = true
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        myDelegate?.srAssetZoomableViewDidLayoutSubviews(self)
    }
    
    /// 设置缩放比例以使显示完整的图像
    public func fitImage(_ isAnimated: Bool = false) {
        let squaredZoomScale = calculateSquaredZoomScale()
        setZoomScale(squaredZoomScale, animated: isAnimated)
    }
    
    /// Re-apply correct scrollview settings if image has already been adjusted in
    /// multiple selection mode so that user can see where they left off.
    public func applyStoredCropPosition(_ scp: SRLibrarySelection) {
        // ZoomScale needs to be set first.
        if let zoomScale = scp.scrollViewZoomScale {
            setZoomScale(zoomScale, animated: false)
        }
        if let contentOffset = scp.scrollViewContentOffset {
            setContentOffset(contentOffset, animated: false)
        }
    }
        
    public func setImage(_ image: UIImage) {
        if photoImageView.isDescendant(of: self) == false {
            addSubview(photoImageView)
            photoImageView.contentMode = .scaleAspectFill
            photoImageView.clipsToBounds = false
        }
        photoImageView.image = image
        setAssetFrame(for: photoImageView, with: image)
    }
    
    private  func setAssetFrame(`for` view: UIView, with image: UIImage) {
        // Reseting the previous scale
        self.minimumZoomScale = 1
        self.zoomScale = 1
        
        // Calculating and setting the image view frame depending on screenWidth
        let w = image.size.width
        let h = image.size.height

        let aspectRatio: CGFloat = width/height
        let aspectRatioImage: CGFloat = w/h

        if aspectRatio > aspectRatioImage {
            view.frame.size.width = width
            view.frame.size.height = width / aspectRatioImage
        } else {
            view.frame.size.width = height * aspectRatioImage
            view.frame.size.height = height
        }
        
        // Centering image view
        centerAssetView()
        
        // Setting new scale
        minimumZoomScale = zoomScale
        self.zoomScale = zoomScale
    }
    
    /// Calculate zoom scale which will fit the image to square
    private  func calculateSquaredZoomScale() -> CGFloat {
        guard let image = assetImageView.image else {
            print("YPAssetZoomableView >>> No image"); return 1.0
        }
        
        var squareZoomScale: CGFloat = 1.0
        let w = image.size.width
        let h = image.size.height
        
        if w > h { // Landscape
            squareZoomScale = (w / h)
        } else if h > w { // Portrait
            squareZoomScale = (h / w)
        }
        
        return squareZoomScale
    }
    
    // Centring the image frame
    private  func centerAssetView() {
        let assetView = photoImageView
        let scrollViewBoundsSize = self.bounds.size
        var assetFrame = assetView.frame
        let assetSize = assetView.frame.size
        assetFrame.origin.x = (assetSize.width > scrollViewBoundsSize.width) ?
            (assetSize.width - scrollViewBoundsSize.width) / 2.0 : 0
        assetFrame.origin.y = (assetSize.height > scrollViewBoundsSize.height) ?
            (assetSize.height - scrollViewBoundsSize.height) / 2.0 : 0.0
        self.contentOffset = assetFrame.origin
    }
    
    func cropImage() -> UIImage? {
        guard let image = photoImageView.image else  { return nil }
        let xCrop = self.contentOffset.x
        let yCrop = self.contentOffset.y
        let widthCrop = self.bounds.size.width
        let heightCrop = self.bounds.size.height
        let scaleRatio = image.size.width / photoImageView.frame.width
        let scaledCropRect = CGRect(x: xCrop * scaleRatio,
                                    y: yCrop * scaleRatio,
                                    width: widthCrop * scaleRatio,
                                    height: heightCrop * scaleRatio)
        if let cgImage = image.toCIImage()?.toCGImage(),
            let imageRef = cgImage.cropping(to: scaledCropRect) {
            let croppedImage = UIImage(cgImage: imageRef)
            return croppedImage
        }
        return nil
    }
}


// MARK: UIScrollViewDelegate Protocol
extension SRCropZoomableView: UIScrollViewDelegate {
    func viewForZooming(in scrollView: UIScrollView) -> UIView? {
        return photoImageView
    }
    
    func scrollViewDidZoom(_ scrollView: UIScrollView) {
        myDelegate?.srAssetZoomableViewScrollViewDidZoom()
    }
    
    func scrollViewDidEndZooming(_ scrollView: UIScrollView, with view: UIView?, atScale scale: CGFloat) {
        guard let view = view, view == photoImageView else { return }
        /// 防止缩小
        if  scale < 1 { self.fitImage(true) }
        myDelegate?.srAssetZoomableViewScrollViewDidEndZooming()
        cropAreaDidChange()
    }
    
    func scrollViewDidEndDragging(_ scrollView: UIScrollView, willDecelerate decelerate: Bool) {
        cropAreaDidChange()
    }
    
    func scrollViewDidEndDecelerating(_ scrollView: UIScrollView) {
        cropAreaDidChange()
    }
}
