//
//  SRAlbumsManager.swift
//  CTFit
//
//  Created by Mac on 2020/11/4.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import Photos
import UIKit

// MARK: - 相册专辑
struct SRAlbum {
    /// 缩略图
    var thumbnail: UIImage?
    /// 标题
    var title: String = ""
    /// 数量
    var numberOfItems: Int = 0
    /// 集合
    var collection: PHAssetCollection?
}

class SRAlbumsManager: NSObject {
    
    private var cachedAlbums: [SRAlbum]?
    
    deinit { unregisterForLibraryChanges() }
    override init() {
        super.init()
        registerForLibraryChanges()
    }
    
    func doAfterFetchAlbums(complete: @escaping (([SRAlbum]) -> Void)) {
        DispatchQueue.global(qos: .userInitiated).async { [weak self] in
            guard let strong = self else { return }
            let albums = strong.fetchAlbums()
            DispatchQueue.main.async {
                complete(albums)
            }
        }
    }
    
    @discardableResult
    func fetchAlbums() -> [SRAlbum] {
        if let cachedAlbums = cachedAlbums { return cachedAlbums }
        
        var albums = [SRAlbum]()
        let options = PHFetchOptions()
        
        let smartAlbumsResult = PHAssetCollection.fetchAssetCollections(with: .smartAlbum,
                                                                        subtype: .any,
                                                                        options: options)
        let albumsResult = PHAssetCollection.fetchAssetCollections(with: .album,
                                                                   subtype: .any,
                                                                   options: options)
        for result in [smartAlbumsResult, albumsResult] {
            result.enumerateObjects({ assetCollection, _, _ in
                var album = SRAlbum()
                album.title = assetCollection.localizedTitle ?? ""
                album.numberOfItems = self.mediaCountFor(collection: assetCollection)
                if album.numberOfItems > 0 {
                    let r = PHAsset.fetchKeyAssets(in: assetCollection, options: nil)
                    if let first = r?.firstObject {
                        let deviceScale = UIScreen.main.scale
                        let targetSize = CGSize(width: 78*deviceScale, height: 78*deviceScale)
                        let options = PHImageRequestOptions()
                        options.isSynchronous = true
                        options.deliveryMode = .opportunistic
                        PHImageManager.default().requestImage(for: first,
                                                              targetSize: targetSize,
                                                              contentMode: .aspectFill,
                                                              options: options,
                                                              resultHandler: { image, _ in
                                                                album.thumbnail = image
                        })
                    }
                    album.collection = assetCollection
                    if SRLibraryConfig.mediaType == .photo {
                        if !(assetCollection.assetCollectionSubtype == .smartAlbumSlomoVideos
                            || assetCollection.assetCollectionSubtype == .smartAlbumVideos) {
                            albums.append(album)
                        }
                    } else {
                        albums.append(album)
                    }
                }
            })
        }
        cachedAlbums = albums
        return albums
    }
    
    func mediaCountFor(collection: PHAssetCollection) -> Int {
        let options = PHFetchOptions()
        options.predicate = SRLibraryConfig.mediaType.predicate()
        let result = PHAsset.fetchAssets(in: collection, options: options)
        return result.count
    }
    
}

extension SRAlbumsManager {
    
    var customerAlbum: SRAlbum? {
        guard let album = cachedAlbums?.first(where: { (album) -> Bool in (album.title == SRPhotoCaptureConfig.albumName) })  else { return nil }
        return album
    }
    
    var defaultAlbum: SRAlbum? {
        if let album = customerAlbum { return album }
        if let album = cachedAlbums?.first { return album }
        return nil
    }
}

extension SRlibraryMediaType {
    func predicate() -> NSPredicate {
        switch self {
        case .photo:
            return NSPredicate(format: "mediaType = %d",
                               PHAssetMediaType.image.rawValue)
        case .video:
            return NSPredicate(format: "mediaType = %d",
                               PHAssetMediaType.video.rawValue)
        case .photoAndVideo:
            return NSPredicate(format: "mediaType = %d || mediaType = %d",
                               PHAssetMediaType.image.rawValue,
                               PHAssetMediaType.video.rawValue)
        }
    }
}

extension SRAlbumsManager: PHPhotoLibraryChangeObserver {
    
    func registerForLibraryChanges() {
        PHPhotoLibrary.shared().register(self)
    }
    
    func unregisterForLibraryChanges() {
        PHPhotoLibrary.shared().unregisterChangeObserver(self)
    }
    
    func photoLibraryDidChange(_ changeInstance: PHChange) {
        cachedAlbums = nil
    }
}


