//
//  SRLibraryView.swift
//  CTFit
//
//  Created by Mac on 2020/11/4.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import Photos
import PhotosUI
import Repeater
import Alamofire

class SRLibraryView: UIView {
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    deinit {
        unregisterForLibraryChanges()
    }
    
    init() {
        super.init(frame: .zero)
        addSubviews()
        onPersission()
    }
        
    private func onPersission() {
        SRPermissionCheckable.Library.doAfterPermissionCheck { [weak self] permission in
            guard let strongSelf = self else { return }
            strongSelf.permissionButton.isHidden = permission
            
            guard permission else { return }
            guard !strongSelf.initialized else { return }
            strongSelf.permissionButton.isHidden = true
            strongSelf.initialize()
            strongSelf.initialized = true
        }
    }
   
    ///: - View customer
    private func addSubviews() {
        let items = [collectionView, permissionButton]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        collectionView.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 1, left: 1, bottom: UIDevice.bottomLayoutGuide+10, right: 1))
        }
        
        permissionButton.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(8)
            make.right.equalToSuperview().offset(-8)
            make.top.equalToSuperview().offset(32)
        }
    }
    
    /// collectionView 布局
    private lazy var flowLayout: UICollectionViewFlowLayout = {
        let layout = UICollectionViewFlowLayout()
        layout.scrollDirection = .vertical
        return layout
    }()
    
    lazy var collectionView: UICollectionView = {
        let collection = UICollectionView(frame: .zero, collectionViewLayout: flowLayout)
        collection.backgroundColor = Colors.Base.background
        collection.showsHorizontalScrollIndicator = true
        collection.isScrollEnabled = true
        collection.dataSource = self
        collection.delegate = self
        collection.register(SRLibraryViewCell.self, forCellWithReuseIdentifier: "SRLibraryViewCell")
        collection.register(SRLimitedEditCell.self, forCellWithReuseIdentifier: "SRLimitedEditCell")
        return collection
    }()
    
    private var cellSize: CGSize {
        let size = UIScreen.main.bounds.width/CGFloat(SRLibraryConfig.numberOfItemsInRow) * UIScreen.main.scale
        return CGSize(width: size, height: size)
    }
    
    private lazy var permissionButton: UIButton = {
        let button = UIButton(style: Stylesheet.View.permissionButton)
        button.setTitle(SRString.Connected.denied_permission_library_message.locastr, for: .normal)
        button.isHidden = true
        button.addTarget(self, action: #selector(toPermission), for: .touchUpInside)
        return button
    }()
    
    
    
    @objc
    private func toPermission() {
        Helper.openPermissionSetting()
    }
    
    // MARK: - Field
    internal var isProcessing = false // 如果视频或图像处于处理状态，为真
    internal var multipleSelectionEnabled = false
    internal var initialized = false
    internal var selection = [SRLibrarySelection]()
    internal var currentlySelectedIndex: Int = -1
    internal let mediaManager = SRLibraryMediaManager()
    internal var latestImageTapped = ""
    
    /// 选取单张照片
    typealias DidPickerPhoto = (UIImage) -> Void
    var didPickerPhoto: DidPickerPhoto?
    
    /// 完成多选
    typealias DidPickerFilish = ([SRLibrarySelection]) -> Void
    var didPickerFilish: DidPickerFilish?
    
    /// 去预览照片
    typealias PreviewAssetClosure = (_ mediaManager: SRLibraryMediaManager, _ defaultIndex: Int) -> Void
    var previewAssetClosure: PreviewAssetClosure?
    
    /// iOS 14  Limited 权限
    var isPermissionLimited: Bool { SRPermissionCheckable.Library.isLimited }
    
    var nrManager: NetworkReachabilityManager? = { NetworkReachabilityManager(host: "www.apple.com") } ()
    
    var netEnable: Bool { nrManager?.isReachable ?? false }
    
    private lazy var requestOptions: PHImageRequestOptions = {
        let options = PHImageRequestOptions()
        //options.version = .current
        options.deliveryMode = .highQualityFormat
        options.isNetworkAccessAllowed = true
        options.resizeMode = .exact
        options.isSynchronous = false // Ok since we're already in a background thread
        return options
    } ()
    
    private var itemCount: Int {
        var count = mediaManager.fetchResultCount
        if isPermissionLimited { count += 1 }
        return count
    }
}

extension SRLibraryView {
    
    func onDidPickerPhoto(_ closure: @escaping DidPickerPhoto) {
        didPickerPhoto = closure
    }
    
    func onDidPickerFilish(_ closure: @escaping DidPickerFilish) {
        didPickerFilish = closure
    }
    
    func onDidPreviewAsset(_ closure: @escaping PreviewAssetClosure) {
        previewAssetClosure = closure
    }
    
    func setAlbum(_ album: SRAlbum) {
        mediaManager.collection = album.collection
        currentlySelectedIndex = -1
        if !multipleSelectionEnabled { selection.removeAll() }
        refreshMediaRequest()
    }
    
    func initialize() {
        setupMultipleLongPressGesture()
        mediaManager.initialize()
        
        guard mediaManager.fetchResult == nil else { return }
        registerForLibraryChanges()
        refreshMediaRequest()
        
        if SRLibraryConfig.defaultMultipleSelection { multipleSelectionButtonTapped() }
        if let preselectedItems = SRLibraryConfig.preselectedItems {
            selection = preselectedItems.compactMap { item -> SRLibrarySelection? in
                var itemAsset: PHAsset?
                switch item {
                case .photo(let photo):
                    itemAsset = photo.asset
                case .video(let video):
                    itemAsset = video.asset
                }
                guard let asset = itemAsset else { return nil }
                /// 负索引将在 collectionView:cellForItemAt: 中更正
                return SRLibrarySelection(index: -1, assetIdentifier: asset.localIdentifier)
            }

            multipleSelectionEnabled = selection.count > 1
            collectionView.reloadData()
        }
    }
    
    private func fetchOptions() -> PHFetchOptions {
        /// 排序条件
        if let userOpt = SRLibraryConfig.options { return userOpt }
        let options = PHFetchOptions()
        options.sortDescriptors = [NSSortDescriptor(key: "creationDate", ascending: true)]
        options.predicate = SRLibraryConfig.mediaType.predicate()
        return options
    }
    
    /// 请求媒体数据
    private func refreshMediaRequest() {
        let options = fetchOptions()
        if let collection = mediaManager.collection {
            mediaManager.fetchResult = PHAsset.fetchAssets(in: collection, options: options)
        } else {
            mediaManager.fetchResult = PHAsset.fetchAssets(with: options)
        }
                
        collectionView.reloadData()
        DispatchQueue.main.async { self.scrollToBottom() }
    }
    
    
    
    private func scrollToTop() {
        collectionView.contentOffset = CGPoint.zero
    }
    
    private func scrollToBottom() {
        let viewHeight = collectionView.height
        let contentHeight = collectionView.contentSize.height
        guard contentHeight > viewHeight else { return }
        let y = contentHeight - viewHeight
        let point = CGPoint(x: 0, y: y)
        collectionView.setContentOffset(point, animated: false)
    }
   
    
    private func changeAsset(_ asset: PHAsset) {
        latestImageTapped = asset.localIdentifier
        if let block = didPickerPhoto, let curView = Helper.currentVc?.view {
            curView.isUserInteractionEnabled = false
            ToastUtils.showActivityCCV()
            DispatchQueue.global().async { [weak self] in
                guard let strong = self else { return }
                strong.mediaManager.imageManager?.fetchImage(for: asset, callback: { (image) in
                    DispatchQueue.main.async {
                        guard let _ = Helper.currentVc as? SRLibraryViewController else {
                            self = nil
                            return
                        }
                        curView.isUserInteractionEnabled = true
                        ToastUtils.hideActivityCCV()
                        guard let hasImage = image else {
                            if !strong.netEnable { ToastUtils.showToast(error: "The network is not reachable.") }
                            return
                        }
                        block(hasImage)
                    }
                })
            }
        }
    }
    
    public func scrollViewDidScroll(_ scrollView: UIScrollView) {
        if scrollView == collectionView {
            // mediaManager.updateCachedAssets(in: collectionView)
        }
    }
}

// MARK: - 多选处理
extension SRLibraryView {
    /// 是否已达到界限
    var isLimitExceeded: Bool { return selection.count >= SRLibraryConfig.maxNumberOfItems }
    
    /// 检查是否可以选择更多的项目，如果没有则出现警告
    func checkLimit() {
        //let isHidden = !isLimitExceeded || multipleSelectionEnabled == false
    }
    
    /// 添加多选长按手势
    private  func setupMultipleLongPressGesture() {
        /// 长按单元格启用多重选择
        let longPressGR = UILongPressGestureRecognizer(target: self, action: #selector(handleLongPress(longPressGR:)))
        longPressGR.minimumPressDuration = 0.5
        collectionView.addGestureRecognizer(longPressGR)
    }
    
    /// 当长按 Cell 时，清除所有先前选择的 Cell
    @objc private func handleLongPress(longPressGR: UILongPressGestureRecognizer) {
        if multipleSelectionEnabled || isProcessing || SRLibraryConfig.maxNumberOfItems <= 1 {
            return
        }
        if longPressGR.state == .began {
            let point = longPressGR.location(in: collectionView)
            guard let indexPath = collectionView.indexPathForItem(at: point) else { return }
            startMultipleSelection(at: indexPath)
        }
    }
    
    private func startMultipleSelection(at indexPath: IndexPath) {
        currentlySelectedIndex = indexPath.row
        multipleSelectionButtonTapped()
    }
    
    @objc
    private  func multipleSelectionButtonTapped() {
        SRPermissionCheckable.Library.doAfterPermissionCheck { [weak self] permission in
            guard let strongSelf = self else { return }
            guard permission else { return }
            strongSelf.showMultipleSelection()
        }
    }
    
    private func showMultipleSelection() {
        if !multipleSelectionEnabled {
            selection.removeAll()
        }
        /// 在使用minNumberOfItems时，防止取消多重选择
        if SRLibraryConfig.minNumberOfItems > 1 && multipleSelectionEnabled {
            return
        }
        multipleSelectionEnabled = !multipleSelectionEnabled
        if multipleSelectionEnabled {
            if selection.isEmpty {
                if let asset = mediaManager.fetchResult?[currentlySelectedIndex] {
                    selection = [SRLibrarySelection(index: currentlySelectedIndex, assetIdentifier: asset.localIdentifier)]
                }
            }
        } else {
            selection.removeAll()
            addToSelection(indexPath: IndexPath(row: currentlySelectedIndex, section: 0))
        }

        collectionView.reloadData()
        checkLimit()
    }
}


// MARK: - Library collection view 视图单元格管理

extension SRLibraryView {
    
    /// 判断是否已经存在是选择
    func isInSelectionPool(indexPath: IndexPath) -> Bool {
        guard let fetchResult = mediaManager.fetchResult,  indexPath.row < fetchResult.count else { return  false}
        return selection.contains(where: { $0.assetIdentifier == fetchResult[indexPath.row].localIdentifier })
    }
    
    /// 从选择中移除单元格 （反选）
    func deselect(indexPath: IndexPath) {
        guard let fetchResult = mediaManager.fetchResult,  indexPath.row < fetchResult.count else { return }
        if let positionIndex = selection.firstIndex(where: { $0.assetIdentifier == fetchResult[indexPath.row].localIdentifier }) {
            selection.remove(at: positionIndex)

            /// 刷新数
            var selectedIndexPaths = [IndexPath]()
            fetchResult.enumerateObjects { [unowned self] (asset, index, _) in
                if self.selection.contains(where: { $0.assetIdentifier == asset.localIdentifier }) {
                    selectedIndexPaths.append(IndexPath(row: index, section: 0))
                }
            }
            collectionView.reloadItems(at: selectedIndexPaths)
            
            /// 将当前选定的图像替换为先前选定的图像
            if let previouslySelectedIndexPath = selectedIndexPaths.last {
                collectionView.deselectItem(at: indexPath, animated: false)
                collectionView.selectItem(at: previouslySelectedIndexPath, animated: false, scrollPosition: [])
                currentlySelectedIndex = previouslySelectedIndexPath.row
            }
            checkLimit()
        }
    }
    
    /// 为选区添加单元格 （正选）
    func addToSelection(indexPath: IndexPath) {
        guard let fetchResult = mediaManager.fetchResult,  indexPath.item < fetchResult.count else { return }
        let asset = fetchResult[indexPath.item]
        selection.append(
            SRLibrarySelection(
                index: indexPath.row,
                assetIdentifier: asset.localIdentifier
            )
        )
        checkLimit()
    }
}

extension SRLibraryView: UICollectionViewDataSource {
    public func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return itemCount
    }
}

extension SRLibraryView: UICollectionViewDelegate {
    func presentLimitedLibraryPicker() {
        guard let currentVc = Helper.currentVc as? SRLibraryViewController,
        let naviVc = currentVc.navigationController  else { return }
        if #available(iOS 14, *) {
            PHPhotoLibrary.shared().presentLimitedLibraryPicker(from: naviVc)
        }
    }
    func formattedStrigFrom(_ timeInterval: TimeInterval) -> String {
        let interval = Int(timeInterval)
        let seconds = interval % 60
        let minutes = (interval / 60) % 60
        return String(format: "%02d:%02d", minutes, seconds)
    }
    
    
    func update(_ cell: SRLibraryViewCell,
                _ asset: PHAsset,
                _ manager: PHCachingImageManager,
                _ cellSize: CGSize,
                _ options: PHImageRequestOptions) {
        cell.imageView.image = nil
        DispatchQueue.global().async {
            // self.isPermissionLimited ? options : nil
            manager.requestImage(for: asset, targetSize: cellSize, contentMode: .aspectFill, options: options) { image, info in
                DispatchQueue.main.async {
                    guard cell.representedAssetIdentifier == asset.localIdentifier else { return }
                    guard let hasImage = image else {return }
                    cell.imageView.image = hasImage
                }
            }
        }
    }
    
    public func collectionView(_ collectionView: UICollectionView,
                               cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        
        
        if isPermissionLimited, indexPath.item == itemCount-1 {
            guard let cell = collectionView.dequeueReusableCell(withReuseIdentifier: "SRLimitedEditCell", for: indexPath) as? SRLimitedEditCell else {
                fatalError("unexpected cell in collection view")
            }
            // Prevent weird animation where thumbnail fills cell on first scrolls.
            UIView.performWithoutAnimation {
                cell.layoutIfNeeded()
            }
            return cell
        } else {
            guard let fetchResult = mediaManager.fetchResult, indexPath.item < fetchResult.count else { return UICollectionViewCell() }
            
            guard let cell = collectionView.dequeueReusableCell(withReuseIdentifier: "SRLibraryViewCell", for: indexPath) as? SRLibraryViewCell else {
                fatalError("unexpected cell in collection view")
            }
            
            let asset = fetchResult[indexPath.item]
            cell.representedAssetIdentifier = asset.localIdentifier
            update(cell, asset, mediaManager.imageManager!, cellSize, requestOptions)
            
            
            let isVideo = (asset.mediaType == .video)
            cell.durationLabel.isHidden = !isVideo
            cell.durationLabel.text = isVideo ? formattedStrigFrom(asset.duration) : ""
            cell.multipleSelectionIndicator.isHidden = !multipleSelectionEnabled
            if multipleSelectionEnabled {
                cell.isSelected = currentlySelectedIndex == indexPath.row
            }
            
            // Set correct selection number
            if let index = selection.firstIndex(where: { $0.assetIdentifier == asset.localIdentifier }) {
                let currentSelection = selection[index]
                if currentSelection.index < 0 {
                    selection[index] = SRLibrarySelection(index: indexPath.row,
                                                          assetIdentifier: currentSelection.assetIdentifier)
                }
                cell.multipleSelectionIndicator.set(number: index + 1) // start at 1, not 0
            } else {
                cell.multipleSelectionIndicator.set(number: nil)
            }
            
            // Prevent weird animation where thumbnail fills cell on first scrolls.
            UIView.performWithoutAnimation {
                cell.layoutIfNeeded()
            }
            return cell
        }
        
        
    }
    
    public func collectionView(_ collectionView: UICollectionView, didSelectItemAt indexPath: IndexPath) {
        if isPermissionLimited, indexPath.item == itemCount-1 {
            presentLimitedLibraryPicker()
            return
        }
        
        guard let fetchResult = mediaManager.fetchResult, indexPath.row < fetchResult.count else { return }
        let previouslySelectedIndexPath = IndexPath(row: currentlySelectedIndex, section: 0)
        currentlySelectedIndex = indexPath.row
        changeAsset(fetchResult[indexPath.row])

        if multipleSelectionEnabled {
            let cellIsInTheSelectionPool = isInSelectionPool(indexPath: indexPath)
            let cellIsCurrentlySelected = previouslySelectedIndexPath.row == currentlySelectedIndex
            if cellIsInTheSelectionPool {
                if cellIsCurrentlySelected {
                    deselect(indexPath: indexPath)
                }
            } else if isLimitExceeded == false {
                addToSelection(indexPath: indexPath)
            }
            collectionView.reloadItems(at: [indexPath])
            collectionView.reloadItems(at: [previouslySelectedIndexPath])
        } else {
            previewAssetClosure?(mediaManager, indexPath.row)
            collectionView.deselectItem(at: indexPath, animated: true)
            
        }
        
    }
    
    public func collectionView(_ collectionView: UICollectionView, shouldSelectItemAt indexPath: IndexPath) -> Bool {
        return isProcessing == false
    }
    
    public func collectionView(_ collectionView: UICollectionView, shouldDeselectItemAt indexPath: IndexPath) -> Bool {
        return isProcessing == false
    }
}

extension SRLibraryView: UICollectionViewDelegateFlowLayout {
    public func collectionView(_ collectionView: UICollectionView,
                               layout collectionViewLayout: UICollectionViewLayout,
                               sizeForItemAt indexPath: IndexPath) -> CGSize {
        let margins = SRLibraryConfig.spacingBetweenItems * CGFloat(SRLibraryConfig.numberOfItemsInRow - 1)
        let width = (collectionView.frame.width - margins) / CGFloat(SRLibraryConfig.numberOfItemsInRow)
        return CGSize(width: width, height: width)
    }

    public func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, minimumLineSpacingForSectionAt section: Int) -> CGFloat {
        return SRLibraryConfig.spacingBetweenItems
    }

    public func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, minimumInteritemSpacingForSectionAt section: Int) -> CGFloat {
        return SRLibraryConfig.spacingBetweenItems
    }
}
