//
//  SRLibraryViewController.swift
//  CTFit
//
//  Created by Mac on 2020/11/4.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import Photos

enum SRLibraryType {
    case device, portrait, dialgroud
}

class SRLibraryViewController: BaseViewController, CustomView {
   
    typealias ViewClass = SRLibraryView
    
    required init(albumsManager: SRAlbumsManager = SRAlbumsManager(), type: SRLibraryType = .portrait) {
        self.albumsManager = albumsManager
        self.type = type
        super.init()
        loadLabraryStyle()
    }
    
    override func loadView() {
        super.loadView()
        view = ViewClass()
    }
    
    override func initNavigationBar() {
        super.initNavigationBar()
        navigationItem.titleView = titieView
        switch type {
        case .device: navigationItem.leftBarButtonItem = backBarItem()
        case .portrait, .dialgroud: navigationItem.leftBarButtonItem = cancelBarItem(); swipe()
        }
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        view.backgroundColor = Colors.black
        
        if #available(iOS 11.0, *) {
            self.customView.collectionView.contentInsetAdjustmentBehavior = .always
        } else {
            self.automaticallyAdjustsScrollViewInsets = false
        }
        
        albumsManager.doAfterFetchAlbums { [weak self] albums in
            guard let strong = self else { return }
            if let text = strong.albumsManager.defaultAlbum?.title { strong.titieView.text = text }

        }
        
        switch type {
        case .portrait, .dialgroud: onDidPickerPhoto()
        case .device:
            setUserAlbum(albumsManager.defaultAlbum)
            onPreviewAsset()
        }
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
    }
    
    override func onBack() {
        switch type {
        case .device: self.navigationController?.popViewController(animated: true)
        case .portrait, .dialgroud: self.navigationController?.dismiss(animated: true, completion: nil)
        }
    }
    
    override func onCancel() {
        switch type {
        case .device: self.navigationController?.popViewController(animated: true)
        case .portrait, .dialgroud: self.navigationController?.dismiss(animated: true, completion: nil)
        }
    }
    
    private func swipe() {
        let swipeRecognizer = UISwipeGestureRecognizer(target: self, action: #selector(onCancel))
        swipeRecognizer.direction = .right
        view.addGestureRecognizer(swipeRecognizer)
    }
    
    // MARK: -  Private fields
    
    private let albumsManager: SRAlbumsManager
    private let type: SRLibraryType
    
    private lazy var titieView: SRAlbumButton = {
        let button = SRAlbumButton()
        button.addTarget(self, action: #selector(toAlbumVC), for: .touchUpInside)
        return button
    } ()
    
    /// 选取单张照片
    typealias DidPickerPhoto = (UIImage) -> Void
    private var didPickerPhoto: DidPickerPhoto?
}

/// 选取单张照片
extension SRLibraryViewController {
    public func onDidPickerPhoto(_ closure: @escaping DidPickerPhoto) {
        didPickerPhoto = closure
    }
    
    private func onDidPickerPhoto() {
        customView.onDidPickerPhoto { [weak self] (image) in
            guard let strong = self else { return }
            strong.pickerPhoto(image)
        }
    }
    
    private func pickerPhoto(_ image: UIImage) {
        if let cropVc = SRCropViewController.enter(image, type: type) {
            cropVc.onDidFinishCropping { [weak self] (image) in
                guard let strong = self else { return }
                strong.didPickerPhoto?(image)
            }
        } else {
            didPickerPhoto?(image)
        }
    }
}

/// 去预览照片
extension SRLibraryViewController {

    private func onPreviewAsset() {
        customView.onDidPreviewAsset { [weak self] (manager, index) in
            guard let strong = self else { return }
            strong.perviewAsset(manager, index)
        }
    }
    
    private func perviewAsset(_ manager: SRLibraryMediaManager, _ index: Int) {
        DispatchQueue.main.async {
            SRLibraryCollectionPreviewController.enter(manager, index)
        }
    }
}

/// Album
extension SRLibraryViewController {
    
    private func loadLabraryStyle() {
        switch type {
        case .device: SRCropConfig.showsCrop = .none
        case .portrait, .dialgroud: SRCropConfig.showsCrop = .rectangle(ratio: 1.0)
        }
    }
    
    private func setUserAlbum(_ album: SRAlbum?) {
        guard let ab = album else { return }
        titieView.text = ab.title
        customView.setAlbum(ab)
    }
    
    @objc private func toAlbumVC() {
        guard let currentVc = Helper.currentVc as? SRLibraryViewController else { return }

        let vc = SRAlbumViewController(albumsManager)
        vc.didSelectAlbum = { [weak self] album in
            guard let strong = self else { return }
            strong.setUserAlbum(album)
            currentVc.navigationController?.dismiss(animated: true, completion: nil)
        }
        
        let naviVc = BaseNavigationController(rootViewController: vc)
        currentVc.navigationController?.present(naviVc, animated: true, completion: nil)
    }
}
