//
//  SRPreviewZoomableView.swift
//  CTFit
//
//  Created by Mac on 2020/11/16.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import UIKit
import Photos

protocol SRPreviewZoomableViewDelegate: AnyObject {
    func srAssetZoomableViewDidLayoutSubviews(_ zoomableView: SRPreviewZoomableView)
    func srAssetZoomableViewScrollViewDidZoom()
    func srAssetZoomableViewScrollViewDidEndZooming()
}

final class SRPreviewZoomableView: UIScrollView {
    public weak var myDelegate: SRPreviewZoomableViewDelegate?
    public var cropAreaDidChange = {}
    public var isVideoMode = false
    public var videoView = SRVideoView()
    public var photoImageView = UIImageView()
    public var squaredZoomScale: CGFloat = 1
    public var minWidth: CGFloat? = nil
    private  var currentAsset: PHAsset?
    
    // 可以是视频预览图像查看或照片图像查看
    public var assetImageView: UIImageView {
        return isVideoMode ? videoView.previewImageView : photoImageView
    }
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        backgroundColor = Colors.dark
        frame.size = CGSize.zero
        clipsToBounds = true
        photoImageView.frame = CGRect(origin: CGPoint.zero, size: CGSize.zero)
        videoView.frame = CGRect(origin: CGPoint.zero, size: CGSize.zero)
        maximumZoomScale = 6.0
        minimumZoomScale = 1
        showsHorizontalScrollIndicator = false
        showsVerticalScrollIndicator = false
        delegate = self
        alwaysBounceHorizontal = true
        alwaysBounceVertical = true
        isScrollEnabled = true
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        myDelegate?.srAssetZoomableViewDidLayoutSubviews(self)
    }
    

    /// 设置缩放比例以使图像符合正方形或显示完整的图像
    //
    /// - Parameters:
    ///   - fit: 如果为真-缩放显示平方。如果假-显示完整。
    public func fitImage(_ fit: Bool, animated isAnimated: Bool = false) {
        squaredZoomScale = calculateSquaredZoomScale()
        if fit {
            setZoomScale(squaredZoomScale, animated: isAnimated)
        } else {
            setZoomScale(1, animated: isAnimated)
        }
    }
    
    /// Re-apply correct scrollview settings if image has already been adjusted in
    /// multiple selection mode so that user can see where they left off.
    public func applyStoredCropPosition(_ scp: SRLibrarySelection) {
        // ZoomScale needs to be set first.
        if let zoomScale = scp.scrollViewZoomScale {
            setZoomScale(zoomScale, animated: false)
        }
        if let contentOffset = scp.scrollViewContentOffset {
            setContentOffset(contentOffset, animated: false)
        }
    }
    
    public func setVideo(_ video: PHAsset,
                         mediaManager: SRLibraryMediaManager,
                         storedCropPosition: SRLibrarySelection?,
                         completion: @escaping () -> Void,
                         updateCropInfo: @escaping () -> Void) {
        mediaManager.imageManager?.fetchPreviewFor(video: video) { [weak self] preview in
            guard let strongSelf = self else { return }
            guard strongSelf.currentAsset != video else { completion() ; return }
            
            if strongSelf.videoView.isDescendant(of: strongSelf) == false {
                strongSelf.isVideoMode = true
                strongSelf.photoImageView.removeFromSuperview()
                strongSelf.addSubview(strongSelf.videoView)
            }
            
            strongSelf.videoView.setPreviewImage(preview)
            
            strongSelf.setAssetFrame(for: strongSelf.videoView, with: preview)
            
            completion()
            
            // Stored crop position in multiple selection
            if let scp173 = storedCropPosition {
                strongSelf.applyStoredCropPosition(scp173)
                //MARK: add update CropInfo after multiple
                updateCropInfo()
            }
        }
        mediaManager.imageManager?.fetchPlayerItem(for: video) { [weak self] playerItem in
            guard let strongSelf = self else { return }
            // guard strongSelf.currentAsset != video else { completion() ; return }
            strongSelf.currentAsset = video
            strongSelf.videoView.loadVideo(playerItem)
            strongSelf.videoView.stop()
        }
    }
    
    public func setImage(_ photo: PHAsset,
                         mediaManager: SRLibraryMediaManager,
                         storedCropPosition: SRLibrarySelection?,
                         completion: @escaping (Bool) -> Void,
                         updateCropInfo: @escaping () -> Void) {
        guard currentAsset != photo else {
            DispatchQueue.main.async { completion(false) }
            return
        }
        currentAsset = photo
        
        mediaManager.imageManager?.fetch(photo: photo) { [weak self] image, isLowResIntermediaryImage in
            guard let strongSelf = self else { return }
            
            if strongSelf.photoImageView.isDescendant(of: strongSelf) == false {
                strongSelf.isVideoMode = false
                strongSelf.videoView.removeFromSuperview()
                strongSelf.videoView.showPlayImage(show: false)
                strongSelf.videoView.deallocate()
                strongSelf.addSubview(strongSelf.photoImageView)
            
                strongSelf.photoImageView.contentMode = .scaleAspectFill
                strongSelf.photoImageView.clipsToBounds = true
            }
            
            strongSelf.photoImageView.image = image
           
            strongSelf.setAssetFrame(for: strongSelf.photoImageView, with: image)
                
            // Stored crop position in multiple selection
            if let scp173 = storedCropPosition {
                strongSelf.applyStoredCropPosition(scp173)
                //MARK: add update CropInfo after multiple
                updateCropInfo()
            }
            completion(isLowResIntermediaryImage)
        }
    }
    
    private  func setAssetFrame(`for` view: UIView, with image: UIImage) {
        // Reseting the previous scale
        self.minimumZoomScale = 1
        self.zoomScale = 1
        
        // Calculating and setting the image view frame depending on screenWidth
        let w = image.size.width
        let h = image.size.height

        let aspectRatio: CGFloat = width/height
        let aspectRatioImage: CGFloat = w/h
        var zoomScale: CGFloat = 1

        if aspectRatio < aspectRatioImage { // Landscape
            view.frame.size.width = width
            view.frame.size.height = width / aspectRatioImage
        } else if aspectRatio > aspectRatioImage {
            view.frame.size.width = height * aspectRatioImage
            view.frame.size.height = height
            if let minWidth = minWidth {
                let k = minWidth / height
                zoomScale = (h / w) * k
            }
        } else { // Square
            view.frame.size.width = width
            view.frame.size.height = height
        }
        
        // Centering image view
        view.center = center
        centerAssetView()
        
        // Setting new scale
        minimumZoomScale = zoomScale
        self.zoomScale = zoomScale
    }
    
    /// Calculate zoom scale which will fit the image to square
    private  func calculateSquaredZoomScale() -> CGFloat {
        guard let image = assetImageView.image else {
            print("YPAssetZoomableView >>> No image"); return 1.0
        }
        
        var squareZoomScale: CGFloat = 1.0
        let w = image.size.width
        let h = image.size.height
        
        if w > h { // Landscape
            squareZoomScale = (w / h)
        } else if h > w { // Portrait
            squareZoomScale = (h / w)
        }
        
        return squareZoomScale
    }
    
    // Centring the image frame
    private  func centerAssetView() {
        let assetView = isVideoMode ? videoView : photoImageView
        let scrollViewBoundsSize = self.bounds.size
        var assetFrame = assetView.frame
        let assetSize = assetView.frame.size
        assetFrame.origin.x = (assetSize.width < scrollViewBoundsSize.width) ?
            (scrollViewBoundsSize.width - assetSize.width) / 2.0 : 0
        assetFrame.origin.y = (assetSize.height < scrollViewBoundsSize.height) ?
            (scrollViewBoundsSize.height - assetSize.height) / 2.0 : 0.0
        assetView.frame = assetFrame
    }
}

// MARK: UIScrollViewDelegate Protocol
extension SRPreviewZoomableView: UIScrollViewDelegate {
    func viewForZooming(in scrollView: UIScrollView) -> UIView? {
        return isVideoMode ? videoView : photoImageView
    }
    
    func scrollViewDidZoom(_ scrollView: UIScrollView) {
        myDelegate?.srAssetZoomableViewScrollViewDidZoom()
        centerAssetView()
    }
    
    func scrollViewDidEndZooming(_ scrollView: UIScrollView, with view: UIView?, atScale scale: CGFloat) {
        guard let view = view, view == photoImageView || view == videoView else { return }
        // prevent to zoom out
        if  scale < squaredZoomScale {
            self.fitImage(false, animated: true)
        }
        myDelegate?.srAssetZoomableViewScrollViewDidEndZooming()
        cropAreaDidChange()
    }
    
    func scrollViewDidEndDragging(_ scrollView: UIScrollView, willDecelerate decelerate: Bool) {
        cropAreaDidChange()
    }
    
    func scrollViewDidEndDecelerating(_ scrollView: UIScrollView) {
        cropAreaDidChange()
    }
}
