//
//  BasePickerView.swift
//  CTFitPro
//
//  Created by Mac on 2020/5/18.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import UIKit
import RxSwift
import RxCocoa
import RxDataSources

enum BasePickerType {
    case single, group
}

class BasePickerView: UIView {
    
    typealias ItemSelectedEvent = (_ rowItem: String, _ component: Int) -> Void
    
    
    public func setItemSelected(_ closure: @escaping ItemSelectedEvent) {
        self.itemSelected = closure
    }
    
    public func resetComponents(_ components: [[String]]) {
        self.components = components;
        switch type {
        case .single: initPickerItem()
        case .group:  initPickerItems()
        }
    }
    
    public func select(_ rows: [String], _ animated: Bool) {
        switch type {
        case .single: selectPickerItem(rows, animated)
        case .group:  selectPickerItems(rows, animated)
        }
    }
    
    public func reload(_ dataSource: [String], _ component: Int, _ rowItem: String, _ animated: Bool) {
        switch type {
        case .single: reloadPickerItem(dataSource, component, rowItem, animated)
        case .group:  reloadPickerItems(dataSource, component, rowItem, animated)
        }
    }

    
    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
           fatalError("init(coder:) has not been implemented")
    }
    
    init(type :BasePickerType = .single) {
        super.init(frame: .zero)
        self.bounds = CGRect(x: 0, y: 0, width: UIDevice.scrWidth, height: 150)
        self.type = type
        self.initialize()
    }

    public convenience init(components: [[String]], type :BasePickerType = .single) {
        self.init(type :type)
        self.resetComponents(components)
    }
    
    private func initialize() {
        switch type {
        case .single: setSubviewsForPickerItem()
        case .group: setSubviewsForPickerItems()
        }
    }
    
    
    // MARK: - ViewCustomer
    private lazy var spacing: CGFloat = 12
    private lazy var pickerItems: [UIPickerView] = [UIPickerView]()
    private lazy var pickersStack: UIStackView = {
        let stack = UIStackView()
        stack.axis = .horizontal
        stack.alignment = .center;
        stack.distribution = .fillEqually
        stack.spacing = spacing
        return stack
    } ()
        
    private func setSubviewsForPickerItems() {
        [pickersStack].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        pickersStack.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: spacing, bottom: 0, right: spacing))
        }
    }
    
    private lazy var SettingPickerView = UIPickerView()
    private func setSubviewsForPickerItem() {
        [SettingPickerView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        SettingPickerView.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: spacing, bottom: 0, right: spacing))
        }
    }
       
    // MARK: - Bind event
    private var type: BasePickerType = .single
    private var itemSelected: ItemSelectedEvent? = nil
    private var components = [[String]]()
    private let disposeBag = DisposeBag()
    private typealias SubjectType = PublishSubject<[[String]]>
    private var subjects: [UIPickerView: SubjectType] = [:]
    private var disposables: [UIPickerView: Disposable] = [:]


    // MARK: - viewPickerAdapter
    private func viewPickerAdapter() -> RxPickerViewViewAdapter<[[String]]> {
        let viewPickerAdapter = RxPickerViewViewAdapter<[[String]]>(
            components: [],
            numberOfComponents: { (dataSource, pickerView, components) -> Int in
                components.count },
            numberOfRowsInComponent: { (dataSource, pickerView, components, component) -> Int in
                return components[component].count },
            viewForRow: { (dataSource, pickerView, components, row, component, view) -> UIView in
                //pickerView.subviews[1].backgroundColor = Colors.white
                //pickerView.subviews[2].backgroundColor = Colors.white
                let label = UILabel(style: Stylesheet.Label.title)
                label.font = .customFont(ofSize: 20)
                label.textAlignment = .center
                label.text = components[component][row]
                return label }
        )
        return viewPickerAdapter
    }
    
    private func unbind(_ pickerView: UIPickerView) {
         guard let disposable = disposables[pickerView] else { return }
        disposable.dispose()
        disposables[pickerView] = nil
    }
}

// MARK: -  多组 UIPickerView（每组只有一列)
extension BasePickerView {
    private func initPickerItems() {
        pickerItems.forEach { (pickerView) in
            unbind(pickerView)
            pickersStack.removeArrangedSubview(pickerView)
            pickerView.removeFromSuperview()
        }
        pickerItems.removeAll()
        subjects.removeAll()
        disposables.removeAll()
        
        for idx in 0..<components.count {
            let pickerView = UIPickerView()
            pickerView.rx.itemSelected.subscribe(onNext: { [weak self] (row, component) in
                guard let strong = self, component==0,  let callback = strong.itemSelected else { return }
                let datasource = strong.components[idx]
                callback(datasource[row], idx)
            }).disposed(by: disposeBag)
            pickerItems.append(pickerView)
            let dataSource = components[idx]

            let subject = SubjectType()
            let disposable = subject.asObservable().bind(to: pickerView.rx.items(adapter: viewPickerAdapter()))
            disposeBag.insert(disposable)
            disposables[pickerView] = disposable
            
            subject.onNext([dataSource])
            subjects[pickerView] = subject
        }
        pickerItems.forEach { (pickerView) in pickersStack.addArrangedSubview(pickerView) }
    }
    
    private func selectPickerItems(_ rows: [String], _ animated: Bool) {
        for idx in 0..<components.count {
            guard idx < rows.count else { return }
            guard idx < pickerItems.count else { return }
            let pickerItem = pickerItems[idx]
            let rowItem = rows[idx]
            let component = components[idx]
            if let row = component.firstIndex(of: rowItem), pickerItem.selectedRow(inComponent: 0) != row {
                pickerItem.selectRow(row, inComponent: 0, animated: animated)
            }
        }
    }
    
    private func reloadPickerItems(_ dataSource: [String], _ component: Int, _ rowItem: String, _ animated: Bool) {
        guard component < components.count else { return }
        let pickerView = pickerItems[component]
        guard let subject = subjects[pickerView] else { return }
        components[component] = dataSource
        subject.onNext([dataSource])
        if let row = dataSource.firstIndex(of: rowItem) {
            pickerView.selectRow(row, inComponent: 0, animated: animated)
        }
    }
}

// MARK: -  单组 UIPickerView
extension BasePickerView {
    
    private func initPickerItem() {
        unbind(SettingPickerView)
        subjects.removeAll()
        disposables.removeAll()

        SettingPickerView.rx.itemSelected.subscribe(onNext: { [weak self] (row, component) in
            guard let strong = self, component<strong.components.count,  let callback = strong.itemSelected else { return }
            let datasource = strong.components[component]
            callback(datasource[row], component)
        }).disposed(by: disposeBag)
        
        let subject = SubjectType()
        let disposable = subject.asObservable().bind(to: SettingPickerView.rx.items(adapter: viewPickerAdapter()))
        disposeBag.insert(disposable)
        disposables[SettingPickerView] = disposable

        subject.onNext(components)
        subjects[SettingPickerView] = subject
    }
    
    private func selectPickerItem(_ rows: [String], _ animated: Bool) {
        for idx in 0..<components.count {
            guard idx < rows.count else { return }
            let rowItem = rows[idx]
            let component = components[idx]
            if let row = component.firstIndex(of: rowItem) {
                SettingPickerView.selectRow(row, inComponent: idx, animated: animated)
            }
        }
    }
    
    private func reloadPickerItem(_ dataSource: [String], _ component: Int, _ rowItem: String, _ animated: Bool) {
        guard component < components.count, let subject = subjects[SettingPickerView] else { return }
        components[component] = dataSource
        subject.onNext(components)
        if let row = dataSource.firstIndex(of: rowItem) {
            SettingPickerView.selectRow(row, inComponent: component, animated: animated)
        }
    }
}

extension RxPickerViewViewAdapter {
    open func pickerView(_ pickerView: UIPickerView, rowHeightForComponent component: Int) -> CGFloat {
        return 34
    }
    
    open func pickerView(_ pickerView: UIPickerView, widthForComponent component: Int) -> CGFloat {
        return 100
    }
}



