//
//  DialogPickerView.swift
//  CTFit
//
//  Created by Mac on 2020/9/12.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import RxSwift
import RxCocoa
import RxDataSources

class DialogPickerView: UIView {
    
    typealias ButtonTapEvent = (_ sender: UIButton) -> Void
    typealias ItemSelectedEvent = (_ rowItem: String, _ component: Int) -> Void

    public func setItemSelected(_ closure: @escaping ItemSelectedEvent) {
        pickerView.setItemSelected(closure)
    }
    
    public func select(_ rows: [String], _ animated: Bool) {
        pickerView.select(rows, animated)
    }
    
    public func reload(_ dataSource: [String], _ component: Int, _ rowItem: String, _ animated: Bool) {
        pickerView.reload(dataSource, component, rowItem, animated)
    }

    
    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
           fatalError("init(coder:) has not been implemented")
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
    }
    
    public init(title: String? = nil,
                components: [[String]],
                leftButtonTitle: String,
                leftButtonTap: @escaping ButtonTapEvent,
                rightButtonTitle: String,
                rightButtonTap: @escaping ButtonTapEvent
                ) {
        super.init(frame: .zero)
        self.title = title
        self.components = components
        self.leftButtonTitle = leftButtonTitle
        self.rightButtonTitle = rightButtonTitle
        self.leftButtonTap = leftButtonTap
        self.rightButtonTap = rightButtonTap
        self.initialize()
    }
    
    
    private func initialize() {
        backgroundColor = Colors.Cell.background
        addSubviews()
        setupConstraints()
    }
    
    private func addSubviews() {
         [titleStack, pickerView, buttonsStack].forEach { view in
             view.translatesAutoresizingMaskIntoConstraints = false
             addSubview(view)
         }
     }

     private func setupConstraints() {
         titleStack.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            let height: CGFloat = title == nil ? 0 : 50
            make.height.equalTo(height)
         }
         pickerView.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(titleStack.snp.bottom)
            make.width.equalTo(UIDevice.scrWidth)
            make.height.equalTo(150)
         }
         buttonsStack.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(spacing)
            make.right.equalToSuperview().offset(-spacing)
            make.top.equalTo(pickerView.snp.bottom)
            make.bottom.equalToSuperview().offset(-UIDevice.bottomLayoutGuide)
         }
     }
    
    
    // MARK: - View customer
    private var components = [[String]]()
    private var title: String? = nil
    private var leftButtonTitle: String? = nil
    private var rightButtonTitle: String? = nil
    private var leftButtonTap: ButtonTapEvent? = nil
    private var rightButtonTap: ButtonTapEvent? = nil
    private let disposeBag = DisposeBag()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.naviTitle)
        label.textAlignment = .center
        label.text = title
        return label
    }()
    
    lazy var pickerView: BasePickerView = {
        let picker = BasePickerView(components: components, type: .group)
        return picker
    }()

    private lazy var titleStack: UIStackView = {
        let stack = UIStackView();
        stack.axis = .horizontal;
        stack.alignment = .center;
        stack.distribution = .fillEqually;
        stack.addArrangedSubview(titleLabel)
        return stack
    } ()
    
    private lazy var spacing: CGFloat = 12
    private lazy var buttonsStack: UIStackView = {
        let stack = UIStackView();
        stack.axis = .horizontal;
        stack.alignment = .center;
        stack.distribution = .fillEqually;
        stack.spacing = spacing
        stack.addArrangedSubview(createButton(leftButtonTitle, leftButtonTap))
        stack.addArrangedSubview(createButton(rightButtonTitle, rightButtonTap))
        return stack
    } ()

    private func createButton(_ text: String?, _ tap: ButtonTapEvent?) ->UIButton {
        let buttom = UIButton(type: .system)
        buttom.apply(Stylesheet.View.dialogButton)
        buttom.setTitleColor(Colors.white, for: .normal)
        buttom.setTitle(text, for: .normal)
        guard let _ = text else { return buttom }
        buttom.isUserInteractionEnabled = true
        buttom.rx.tap.subscribe(onNext: { _ in
            if let hasTap = tap { hasTap(buttom) }
        }).disposed(by: disposeBag)
        return buttom
    }
}
