//
//  SliderSegmented.swift
//  CTFit
//
//  Created by Mac on 2020/5/28.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit

typealias SliderSegmentedDisSelectedClosure = (_ idx: Int) -> Void

class SliderSegmented: UIView {
    
    var switchEnable = true
    
    /// 回调闭包
    var didSelectedColuser: SliderSegmentedDisSelectedClosure?
    
    /// 标题
    var viewControllers: [UIViewController] = [UIViewController]() {
        didSet {
            collectionView.reloadData()
            selectIndex = 0
        }
    }
    
    /// 选中位置
    var selectIndex: Int? = nil {
        didSet {
            guard selectIndex != oldValue else { return }
            updateCell(oldValue)
        }
    }
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        initSubviews()
        snp.makeConstraints { (make) in
            make.width.equalTo(kDefaultItemWidth*2)
            make.height.equalTo(44)
        }
    }

    
    // MARK: 外部字段
    
    /// 字体大小
    var kFontSize: CGFloat = 18
    
    /// 最大放大倍数
    var kMaxScale: CGFloat = 1
   
    
    // MARK: 常量字段
    
    /// 间隔
    private let kMargin: CGFloat = 0
    
    /// 滑块线条高度
    private let kLineHeight: CGFloat = 0
    
    /// 更多按钮 宽度
    private let kDefaultItemWidth: CGFloat = UIDevice.scrWidth/2
    
    /// 正常时颜色
    private let normalColor: UIColor = Colors.gray
       
    /// 选中时颜色
    private let selectedColor: UIColor = Colors.white
    
    // MARK: - 控件
    
    /// collection 布局
    lazy var flowLayout: UICollectionViewFlowLayout = {
        let layout = UICollectionViewFlowLayout()
        layout.scrollDirection = .horizontal
        layout.sectionInset = UIEdgeInsets(top: 0, left: kMargin, bottom: 0, right: kMargin)
        return layout
    }()
    
    /// collectionView
    lazy var collectionView: UICollectionView = {
        let collection = UICollectionView(frame: .zero, collectionViewLayout: flowLayout)
        collection.backgroundColor = .clear
        collection.showsHorizontalScrollIndicator = false
        collection.isScrollEnabled = false
        collection.register(SliderCollectionCell.self, forCellWithReuseIdentifier: "SliderCollectionCell")
        collection.dataSource = self
        collection.delegate = self
        return collection
    }()
    
    /// bottomView
    lazy var bottomView: UIView = {
        let view = UIView()
        return view
    }()
    
    /// sliderLine
    lazy var sliderLinde: UILabel = {
        let line = UILabel()
        line.backgroundColor = selectedColor
        return line
    }()
    
    // MARK: - method
    
    private func initSubviews() {
        addSubview(collectionView)
        addSubview(bottomView)
        bottomView.addSubview(sliderLinde)
        
        collectionView.snp.makeConstraints { (make) in
            make.edges.equalTo(self).inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
        bottomView.snp.makeConstraints { (make) in
            make.left.bottom.right.equalToSuperview()
            make.height.equalTo(kLineHeight)
        }
        
        sliderLinde.snp.makeConstraints { (make) in
            make.width.equalTo(kDefaultItemWidth)
            make.height.equalTo(kLineHeight)
            make.center.equalToSuperview()
        }
    }

    
    /// itemWidthForIndex
    private func itemWidthForIndex(_ index: Int) -> CGFloat{
        return kDefaultItemWidth
    }
    
    /// 更新 Cell
    private func updateCell(_ oldIndex: Int?) {
        if let index = oldIndex, let cell = collectionView.cellForItem(at: IndexPath(row: index, section: 0)) as? SliderCollectionCell {
            cell.label.font = .customFont(ofSize: kFontSize)

            cell.label.textColor = normalColor
            cell.transform = CGAffineTransform(scaleX: 1.0, y: 1.0);
        }
        
        if let index = selectIndex, let cell = collectionView.cellForItem(at: IndexPath(row: index, section: 0)) as? SliderCollectionCell {
            cell.label.font = .mediumCustomFont(ofSize: kFontSize)
            cell.label.textColor = selectedColor
            cell.transform = CGAffineTransform(scaleX: kMaxScale, y: kMaxScale);
            collectionView.scrollToItem(at: IndexPath(row: index, section: 0), at: .centeredHorizontally, animated: false)
        }
    }
    
}



// MARK: - UICollectionViewDelegateFlowLayout

extension SliderSegmented: UICollectionViewDelegateFlowLayout {
    
    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, sizeForItemAt indexPath: IndexPath) -> CGSize {
        return CGSize(width: itemWidthForIndex(indexPath.row) + kMargin, height: collectionView.bounds.size.height)
    }
    
    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, minimumLineSpacingForSectionAt section: Int) -> CGFloat {
        return kMargin
    }
    
    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, minimumInteritemSpacingForSectionAt section: Int) -> CGFloat {
        return kMargin
    }
}

// MARK: - UICollectionViewDataSource

extension SliderSegmented: UICollectionViewDataSource {
    
    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return viewControllers.count
    }
    
    func collectionView(_ collectionView: UICollectionView, cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        let cell: SliderCollectionCell = collectionView.dequeueReusableCell(withReuseIdentifier: "SliderCollectionCell", for: indexPath) as! SliderCollectionCell
        if indexPath.row>=0, indexPath.row < viewControllers.count{
            let vc = viewControllers[indexPath.row]
            cell.label.text = vc.navigationItem.title
        } else {
            cell.label.text = ""
        }
        
        cell.label.font = (indexPath.row == selectIndex) ? .mediumCustomFont(ofSize: kFontSize): .customFont(ofSize: kFontSize)
        cell.label.textColor = (indexPath.row == selectIndex) ? selectedColor : normalColor
        let scale: CGFloat = (indexPath.row == selectIndex) ? kMaxScale : 1.0
        cell.transform = CGAffineTransform(scaleX: scale, y: scale)
        
        return cell
    }
}

// MARK: - UICollectionViewDelegate


extension SliderSegmented: UICollectionViewDelegate {
    
    func collectionView(_ collectionView: UICollectionView, didSelectItemAt indexPath: IndexPath) {
        guard indexPath.row>=0, indexPath.row < viewControllers.count, self.selectIndex != indexPath.row, switchEnable else { return }
        guard let coluser = didSelectedColuser else { return }
        coluser(indexPath.row)
    }
}
