//
//  SliderSwitch.swift
//  CTFit
//
//  Created by Mac on 2020/5/28.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit

class SliderSwitch: UIView {
    
    /// 页面
    var viewControllers: [UIViewController] = [UIViewController]() {
        didSet {
            sliderSegmented.viewControllers = viewControllers
            switchToIndex(0, animated: false)
        }
    }
    
    /// 选中位置
    var selectIndex: Int? = nil

    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
       
    private init() {
        super.init(frame: .zero)
        initSubviews()
    }
    
    convenience init(superController: UIViewController) {
        self.init()
        superController.addChild(pageController)
    }

       
    // MARK: - internal
    
    let sliderSegmented = SliderSegmented()
    
    let pageController: UIPageViewController = {
        let controller = UIPageViewController(transitionStyle: .scroll, navigationOrientation: .horizontal, options: [.interPageSpacing: 0])
        
        return controller
    }()

    
    // MARK: - method
    
    private func initSubviews() {
        addSubview(pageController.view)
        pageController.dataSource = self
        pageController.delegate = self
        addScrollViewDelegate()
        pageController.view.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
        sliderSegmented.didSelectedColuser = { [weak self] (idx) in
            guard let strong = self else { return }
            strong.switchToIndex(idx, animated: true)
        }
    }
    

    
    private func switchToIndex(_ index: Int, animated: Bool) {
        guard index >= 0, index < viewControllers.count else { return }
        if animated { sliderSegmented.switchEnable = false }
        let direction: UIPageViewController.NavigationDirection =  (index >= selectIndex ?? 0) ? .forward : .reverse
        pageController.setViewControllers([viewControllers[index]], direction: direction, animated: animated) { [weak self](_) in
            guard let strong = self else { return }
            strong.selectIndex = index
            strong.sliderSegmented.selectIndex = index
            if animated { strong.sliderSegmented.switchEnable = true }
        }
    }
    
}


// MARK: - UIPageViewControllerDataSource

extension SliderSwitch: UIPageViewControllerDataSource {
    func pageViewController(_ pageViewController: UIPageViewController, viewControllerBefore viewController: UIViewController) -> UIViewController? {
        guard let index = viewControllers.firstIndex(of: viewController), index > 0 else { return nil }
        return viewControllers[index-1]
    }
    
    func pageViewController(_ pageViewController: UIPageViewController, viewControllerAfter viewController: UIViewController) -> UIViewController? {
        guard let index = viewControllers.firstIndex(of: viewController), index < viewControllers.count-1 else { return nil }
        return viewControllers[index+1]
    }
}

// MARK: - UIPageViewControllerDelegate

extension SliderSwitch: UIPageViewControllerDelegate {
    
    func pageViewController(_ pageViewController: UIPageViewController, willTransitionTo pendingViewControllers: [UIViewController]) {
        // UILog.v("willTransitionTo")
    }
    
    func pageViewController(_ pageViewController: UIPageViewController, didFinishAnimating finished: Bool, previousViewControllers: [UIViewController], transitionCompleted completed: Bool) {
        // UILog.v("didFinishAnimating finished: \(finished), completed: \(completed)")
        if completed {
            guard let viewController = pageViewController.viewControllers?.first,
                let index = viewControllers.firstIndex(of: viewController) else { return }
            // UILog.v("index: \(index), viewController:\(viewControllers[index])")
            sliderSegmented.selectIndex = index
            selectIndex = index;
        }
    }
}

extension SliderSwitch: UIScrollViewDelegate {
    
    private func addScrollViewDelegate() {
        pageController.view.subviews.forEach { (e) in
            if let scrollView = e as? UIScrollView {
                scrollView.delegate = self
            }
        }
    }
    
    private func cancelScrollViewDelegate() {
        pageController.view.subviews.forEach { (e) in
            if let scrollView = e as? UIScrollView {
                scrollView.delegate = nil
            }
        }
    }


    func scrollViewWillBeginDragging(_ scrollView: UIScrollView) {
        // UILog.v("scrollViewWillBeginDragging")
        self.sliderSegmented.isUserInteractionEnabled = false
    }


    func scrollViewDidEndDecelerating(_ scrollView: UIScrollView) {
        // UILog.v("scrollViewDidEndDecelerating")
        self.sliderSegmented.isUserInteractionEnabled = true
    }
}
