//
//  SupportUtils.swift
//  CTFit
//
//  Created by Mac on 2021/1/30.
//  Copyright © 2021 jpaxh. All rights reserved.
//

import Foundation
import UIKit

enum TargetType: String {
    case ctfit      = "com.jpaxh.CTFit"
    case manspro    = "com.shirajo.manspro"
    case mtkband    = "com.shirajo.mtkband"
    case abyx       = "com.abyxfit.abyxfitsmart"
    case wbfit      = "com.weltbild.weltbildfit"
    case omniwatch  = "com.oaxis.omniwatch"
    case smartwatch = "com.sww.smartwatch"
    case tinwoo     = "cn.shirajo.tinwoo"
    case yirsur     = "com.shirajo.yirsur"
    case wrangler   = "com.shirajo.wrangler"
    
    static let value = TargetType(rawValue: AppUtils.shared.bundleId) ?? .ctfit
}

class SupportUtils {
    
    private static let shared_z: SupportUtils_Z = SupportUtils_Z()
    private static let shared_s: SupportUtils_S = SupportUtils_S()

    static let targetType = TargetType.value

    static func support(type: BDProtocolType, name: String) -> Bool {
        var filters: [SupportModel] = []
        switch type {
        case .series_z: filters = shared_z.filters
        case .series_s: filters = shared_s.filters
        }
        guard filters.contains(where: { (model) -> Bool in
            name.hasPrefix(model.prefixName)
        }) else { return false }
        return true
    }
    
    // 使用固件升级功能
    static var upgradeFeature: Bool {
        switch targetType {
        case .tinwoo, .yirsur: return true
        default: return false
        }
    }
    
    // 使用启动页隐私许可
    static public var privacyEnable: Bool {
        if UIDevice.idiom == .pad { return false }
        switch targetType {
        case .omniwatch, .abyx, .wrangler: return true
        default: return false
        }
    }
}


fileprivate struct SupportModel {
    var prefixName: String
    var prefixNameDfu: String
    var model: String? = nil
}

// MARK: - SupportUtils_Z

fileprivate class SupportUtils_Z {
    
    private let targetType = TargetType.value

    // MARK: - 蓝牙地址支持分类
    
    private lazy var filters_dfu: [SupportModel] = {
        [ SupportModel(prefixName: "DFU_", prefixNameDfu: "DFU_") ]
    } ()

    private lazy var filters_ctfit: [SupportModel] = {
        [ SupportModel(prefixName: "Z6", prefixNameDfu: "DFU_Z6"),
          SupportModel(prefixName: "Z7", prefixNameDfu: "DFU_Z7"),
          SupportModel(prefixName: "Z8", prefixNameDfu: "DFU_Z8"),
          SupportModel(prefixName: "Z9", prefixNameDfu: "DFU_Z9"),
          SupportModel(prefixName: "Z9P", prefixNameDfu: "DFU_Z9P"),
          SupportModel(prefixName: "Z11", prefixNameDfu: "DFU_Z11"),
          SupportModel(prefixName: "Z12", prefixNameDfu: "DFU_Z12"),
          SupportModel(prefixName: "Z15", prefixNameDfu: "DFU_Z15"),
          SupportModel(prefixName: "Z20", prefixNameDfu: "DFU_Z20"),
          SupportModel(prefixName: "Z21", prefixNameDfu: "DFU_Z21"),
          SupportModel(prefixName: "Z22", prefixNameDfu: "DFU_Z22"),
          
          SupportModel(prefixName: "Z12C", prefixNameDfu: "DFU_Z12C"),
          SupportModel(prefixName: "Z20W", prefixNameDfu: "DFU_Z20W"),
          SupportModel(prefixName: "Z21W", prefixNameDfu: "DFU_Z21W"),
        ]
    } ()
    
    private lazy var filters_ctfit_more: [SupportModel] = {
        [
            /// : - 用户
            // SOUCCESS
            SupportModel(prefixName: "AS1", prefixNameDfu: "DFU_Z12C"),
            // Fonex
            SupportModel(prefixName: "ENERGYFP", prefixNameDfu:"DFU_Z12"),
            // JASANI
            SupportModel(prefixName: "LEVIN", prefixNameDfu:"DFU_Z12"),
            // 研胜
            SupportModel(prefixName: "MY9521BK" , prefixNameDfu: "DFU_Z9"),
            SupportModel(prefixName: "VHA-17"   , prefixNameDfu: "DFU_Z11"),
            // DEF
            SupportModel(prefixName: "NK-PA3124-HR" , prefixNameDfu: "DFU_Z6"),
            SupportModel(prefixName: "NK-SW1907"    , prefixNameDfu: "DFU_Z7"),
            SupportModel(prefixName: "NK-SW1911"    , prefixNameDfu: "DFU_Z11"),
            SupportModel(prefixName: "ALLURE"       , prefixNameDfu: "DFU_Z7"),
            SupportModel(prefixName: "ROGUE"        , prefixNameDfu: "DFU_Z11"),
            // FICUS
            SupportModel(prefixName: "CT-FIT-12", prefixNameDfu: "DFU_Z12"),
            SupportModel(prefixName: "CT-FIT-15", prefixNameDfu: "DFU_Z15"),
            // 海威特
            SupportModel(prefixName: "H1103A", prefixNameDfu: "DFU_Z12"),
            SupportModel(prefixName: "R1", prefixNameDfu: "DFU_Z12C"),
            SupportModel(prefixName: "R2", prefixNameDfu: "DFU_Z20"),
            SupportModel(prefixName: "RS01", prefixNameDfu: "DFU_Z12"),
            SupportModel(prefixName: "M9005W", prefixNameDfu: "DFU_Z20W"),
        ]
    } ()

    private lazy var filters_mtkband: [SupportModel] = {
        [ SupportModel(prefixName: "RT984", prefixNameDfu: "DFU_Z6"),
          SupportModel(prefixName: "RT823", prefixNameDfu: "DFU_Z9"),
          SupportModel(prefixName: "RT824", prefixNameDfu: "DFU_Z15"),
          SupportModel(prefixName: "RT023", prefixNameDfu: "DFU_Z12"),
          SupportModel(prefixName: "RT024", prefixNameDfu: "DFU_Z12C"),
        ]
    } ()
    
    private lazy var filters_wbfit: [SupportModel] = {
        [ SupportModel(prefixName: "WELTBILD FIT", prefixNameDfu: "DFU_Z7"),
          SupportModel(prefixName: "AS1", prefixNameDfu: "DFU_Z12C"),
        ]
    } ()
    
    private lazy var filters_omniwatch: [SupportModel] = {
        [ SupportModel(prefixName: "OmniWatch", prefixNameDfu: "DFU_Z9") ]
    } ()
    
    private lazy var filters_smartwatch: [SupportModel] = {
        [ SupportModel(prefixName: "SW 294", prefixNameDfu: "DFU_Z7") ]
    } ()
    
    private lazy var filters_abyx: [SupportModel] = {
        [ SupportModel(prefixName: "ABYX FIT PRO", prefixNameDfu: "DFU_Z7"),
          SupportModel(prefixName: "Abyx Fit Pro", prefixNameDfu: "DFU_Z7"),
        ]
    } ()
    
    private lazy var filters_tinwoo: [SupportModel] = {
        [ SupportModel(prefixName: "T12", prefixNameDfu: "DFU_T12"),
          SupportModel(prefixName: "T20", prefixNameDfu: "DFU_T20"),
          SupportModel(prefixName: "T21", prefixNameDfu: "DFU_T21"),
          SupportModel(prefixName: "T20W", prefixNameDfu: "DFU_T20W"),
          SupportModel(prefixName: "T21W", prefixNameDfu: "DFU_T21W"),
          
          SupportModel(prefixName: "T21L",  prefixNameDfu: "DFU_L21C2"),
          SupportModel(prefixName: "T21WL", prefixNameDfu: "DFU_L21C1"),
          SupportModel(prefixName: "R12",  prefixNameDfu: "DFU_R12C1"),
          SupportModel(prefixName: "R20",  prefixNameDfu: "DFU_R20C2"),
          SupportModel(prefixName: "R20W", prefixNameDfu: "DFU_R20C1"),
        ]
    } ()
    
    private lazy var filters_yirsur: [SupportModel] = {
        [ SupportModel(prefixName: "Y12", prefixNameDfu: "DFU_Y12"),
          SupportModel(prefixName: "Y20", prefixNameDfu: "DFU_Y20"),
          SupportModel(prefixName: "Y21", prefixNameDfu: "DFU_Y21"),
          SupportModel(prefixName: "Y21W", prefixNameDfu: "DFU_Y21W"),
        ]
    } ()
    
    private lazy var filters_wrangler: [SupportModel] = {
        [ SupportModel(prefixName: "WRZ12", prefixNameDfu: "DFU_ZCCD2"),
          SupportModel(prefixName: "WRZ15", prefixNameDfu: "DFU_Z15C4"),
          SupportModel(prefixName: "WRZ20", prefixNameDfu: "DFU_Z20D1"),
        ]
    } ()
    
    private lazy var filters_all: [SupportModel] = {
        return filters_dfu +
            filters_ctfit + filters_ctfit_more +
            filters_mtkband +
            filters_wbfit +
            filters_omniwatch +
            filters_smartwatch +
            filters_abyx +
            filters_tinwoo +
            filters_yirsur +
            filters_wrangler
    } ()
    
    fileprivate lazy var filters: [SupportModel] = {
        var results: [SupportModel] = []
        #if DEBUG
        results += filters_all
        #else
        results += filters_dfu
        switch targetType {
        case .ctfit: results = filters_all
        case .manspro: results += filters_ctfit
        case .mtkband: results += filters_mtkband
        case .abyx: results += filters_abyx
        case .wbfit: results += filters_wbfit
        case .omniwatch: results += filters_omniwatch
        case .smartwatch: results += filters_smartwatch
        case .tinwoo: results += filters_tinwoo + filters_yirsur + filters_ctfit
        case .yirsur: results += filters_tinwoo + filters_yirsur + filters_ctfit
        case .wrangler: results += filters_wrangler
        }
        #endif
        return results
    } ()
}

// MARK: - SupportUtils_S

fileprivate class SupportUtils_S {
    
    private let targetType = TargetType.value

    // MARK: - 蓝牙地址支持分类
    
    private lazy var filters_ctfit: [SupportModel] = {
        [ SupportModel(prefixName: "S2", prefixNameDfu: "S2"),
          SupportModel(prefixName: "S3", prefixNameDfu: "S3"),
          SupportModel(prefixName: "S7", prefixNameDfu: "S7"),
          SupportModel(prefixName: "SB S7",     prefixNameDfu: "SB S7"),
          SupportModel(prefixName: "watchABC",  prefixNameDfu: "watchABC"),
          SupportModel(prefixName: "SMARTBAND", prefixNameDfu: "SMARTBAND"),
        ]
    } ()
    
    private lazy var filters_mtkband: [SupportModel] = {
        [ SupportModel(prefixName: "RT729", prefixNameDfu: "RT729"),
          SupportModel(prefixName: "RT822", prefixNameDfu: "RT822"),
        ]
    } ()
    
    private lazy var filters_smartwatch: [SupportModel] = {
        [ SupportModel(prefixName: "SW294", prefixNameDfu: "SW294") ]
    } ()
    
    private lazy var filters_abyx: [SupportModel] = {
        [ SupportModel(prefixName: "Abyx-Smart3", prefixNameDfu: "Abyx-Smart3") ]
    } ()
    
    private lazy var filters_tinwoo: [SupportModel] = {
        [ SupportModel(prefixName: "T7", prefixNameDfu: "T7") ]
    } ()
    
    
    
    private lazy var filters_all: [SupportModel] = {
        return filters_ctfit +
            filters_mtkband +
            filters_smartwatch +
            filters_abyx +
            filters_tinwoo
    } ()
    
    fileprivate lazy var filters: [SupportModel] = {
        var results: [SupportModel] = []
        #if DEBUG
        results += filters_all
        #else
        switch targetType {
        case .ctfit: results = filters_all
        case .mtkband: results = filters_mtkband
        case .abyx: results = filters_abyx
        case .smartwatch: results = filters_smartwatch
        case .tinwoo: results = filters_tinwoo
        default : break
        }
        #endif
        return results
    } ()
}
