import UIKit
import MJRefresh
import RxSwift

class BaseTableView: UIView {

    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(style: UITableView.Style = .grouped) {
        super.init(frame: .zero)
        self.backgroundColor = Colors.Cell.background
        self.style = style
        addSubviews()
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        if autoBounces {
            DispatchQueue.main.asyncAfter(deadline: .now()+0.1) {
                self.tableView.bounces = self.tableView.contentSize.height > self.tableView.height ? true : false
            }
        }
    }
    
    private func addSubviews() {
        [tableView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        tableView.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            if #available(iOS 11.0, *) {
                make.top.equalTo(self.safeAreaLayoutGuide.snp.top)
                make.bottom.equalTo(self.safeAreaLayoutGuide.snp.bottom)
            } else {
                make.top.equalTo(self.layoutMarginsGuide.snp.top)
                make.bottom.equalTo(self.layoutMarginsGuide.snp.bottom)
            }
        }
    }
    
    func reloadForRecord() {
        [groundImage].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            insertSubview(view, at: 0)
        }
        groundImage.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: -38, left: 0, bottom: 0, right: 0))
        }
        tableView.snp.remakeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(38)
            if #available(iOS 11.0, *) {
                make.bottom.equalTo(self.safeAreaLayoutGuide.snp.bottom)
            } else {
                make.bottom.equalTo(self.layoutMarginsGuide.snp.bottom)
            }
        }
    }
    
    func reloadForRecordHeart() {
        [groundImage].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            insertSubview(view, at: 0)
        }
        groundImage.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
    }
    
    // MARK: - View customer
    
    typealias Anything = () -> Void
        
    let disposeBag = DisposeBag()
    
    var style: UITableView.Style = .grouped
    
    var autoBounces: Bool = true
    
    lazy var groundImage: UIImageView = {
        let view = UIImageView()
        view.contentMode = .scaleAspectFill
        view.backgroundColor = Colors.Base.background
        view.clipsToBounds = true
        return view
    }()

    
    lazy var tableView: UITableView = {
        let tableView = UITableView(frame: .zero, style: style)
        tableView.estimatedRowHeight = 44
        tableView.estimatedSectionHeaderHeight = 0;
        tableView.estimatedSectionFooterHeight = 0;
        tableView.backgroundColor = UIColor.clear
        tableView.clipsToBounds = true
        tableView.showsVerticalScrollIndicator = false
        tableView.showsHorizontalScrollIndicator = false
        return tableView
    }()

    private lazy var mj_header: MJRefreshHeader = {
        let header = MJRefreshNormalHeader();
        header.lastUpdatedTimeLabel?.isHidden = true;
        header.setTitle(SRString.Searched.search_device.locastr, for: .idle)
        header.setTitle(SRString.Searched.search_device.locastr, for: .pulling)
        header.setTitle(SRString.Searched.searching.locastr, for: .refreshing)
        return header;
    }()
    
    // MARK: - Methods
    
    func setRefreshHeader(_ action: Anything? = nil) {
        autoBounces = false
        tableView.mj_header = mj_header;
        mj_header.rx.refreshing.asDriver()
            .drive(onNext: { _ in (action ?? {})() })
            .disposed(by: disposeBag)
        mj_header.beginRefreshing()
    }

    func endRefreshing() {
        mj_header.endRefreshing()
    }

    func setTableView(dataSource: UITableViewDataSource, delegate: UITableViewDelegate) {
        self.tableView.dataSource = dataSource
        self.tableView.delegate = delegate
    }

    func register(cellType: UITableViewCell.Type, forCellReuseIdentifier identifier: String) {
        tableView.register(cellType, forCellReuseIdentifier: identifier)
    }

    func indexPath(for cell: UITableViewCell) -> IndexPath? {
        return tableView.indexPath(for: cell)
    }

    func refreshTableView() {
        DispatchQueue.main.async { [weak self] in
            self?.tableView.reloadData()
        }
    }
}

extension UIView{
    func belongController()->UIViewController?{
        for view in sequence(first: self.superview, next: {$0?.superview}){
            if let responder = view?.next {
                if responder.isKind(of: UIViewController.self){
                    return responder as? UIViewController
                }
            }
        }
        return nil
    }
}
