//
//  AppUtils.swift
//  CTFit
//
//  Created by Mac on 2020/7/20.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import Foundation
import HandyJSON
import MessageUI
import CoreTelephony
import SSZipArchive

extension Bundle {
    open var appIcon: UIImage? {
        if let icons = infoDictionary?["CFBundleIcons"] as? [String: Any],
            let primaryIcon = icons["CFBundlePrimaryIcon"] as? [String: Any],
            let iconFiles = primaryIcon["CFBundleIconFiles"] as? [String],
            let lastIcon = iconFiles.last {
            return UIImage(named: lastIcon)
        }
        return nil
    }
    open var displayName: String {
        if let name = infoDictionary?["CFBundleDisplayName"] as? String { return name }
        if let name = infoDictionary?["CFBundleName"] as? String { return name }
        return "CTFit"
    }
    open var version: String? {
        if let ver = infoDictionary?["CFBundleShortVersionString"] as? String { return ver }
        return nil
    }
    open var bundleVersion: String? {
        if let ver = infoDictionary?["CFBundleVersion"] as? String { return ver }
        return nil
    }
    
    open var backgroundModes: [String] {
        if let modes = infoDictionary?["UIBackgroundModes"] as? [String] { return modes }
        return [];
    }
    
    open var identifier: String? { bundleIdentifier }
}

class AppUtils: NSObject, BDModelProtocol {
    
    static let shared: AppUtils = AppUtils()
        
    var bundleId: String = Bundle.main.bundleIdentifier ?? ""

    var name: String = Bundle.main.displayName
    
    var version: String = Bundle.main.version ?? ""
    
    var buildVersion: String = Bundle.main.bundleVersion ?? ""
        
    var iosVersion: String = "IOS \(UIDevice.current.systemVersion)"

    var lanIdentity: String = NELanguage.latestSystemIdentity() ?? ""
    
    var uuid: String = {
        var text = UIDevice.current.identifierForVendor?.uuidString ?? ""
        text = text.replacingOccurrences(of: "-", with: "")
        return text
    }()

    var carrierName: String = AppUtils.carrierName()

    var deviceModel: String = AppUtils.deviceModel()
    
    // MARK: - Initialize
    required public override init() { }
    
    override class func description() -> String {
        shared.toJSONString(prettyPrint: true)!
    }
}


// MARK: - 意见反馈 MFMailComposeViewControllerDelegate

extension AppUtils: MFMailComposeViewControllerDelegate {
    
    typealias CompletedClosure = (_ state: Bool)->Void
    
    fileprivate static let kLoggerZipName = "logger.zip"
    
    fileprivate func doucmentPath() -> String? {
        return NSSearchPathForDirectoriesInDomains(.documentDirectory, .userDomainMask, true).first;
    }
    
    fileprivate func cachesPath() -> String? {
        return NSSearchPathForDirectoriesInDomains(.cachesDirectory, .userDomainMask, true).first;
    }
    
    fileprivate func zipArchivePath() -> String? {
        guard let cachesPath = cachesPath() as NSString? else { return nil }
        let path = cachesPath.appendingPathComponent(AppUtils.kLoggerZipName)
        let fileManager = FileManager.default;
        let fileExists = fileManager.fileExists(atPath: path) ;
        if (fileExists) {
            return path;
        } else {
            return nil;
        }
    }
    
    fileprivate func zipArchive(_ closure:@escaping CompletedClosure) {
        
        guard let doucmentPath = doucmentPath() as NSString? else { closure(false); return }
        
        let loggerDirectory = doucmentPath.appendingPathComponent("Logger")
        let fileManager = FileManager.default;
        var directory: ObjCBool = ObjCBool(false)
        let exists = fileManager.fileExists(atPath: loggerDirectory, isDirectory: &directory)
        
        guard exists, directory.boolValue else { closure(false); return }
        guard let cachesPath = cachesPath() as NSString? else { closure(false); return }
        
        let zipPath = cachesPath.appendingPathComponent(AppUtils.kLoggerZipName)
        
        DispatchQueue.global().async {
            SSZipArchive.createZipFile(atPath: zipPath, withContentsOfDirectory: loggerDirectory)
            // SSZipArchive.createZipFile(atPath: zipPath, withContentsOfDirectory: loggerDirectory, withPassword: "shirajo@pwd")
            DispatchQueue.main.async {
                closure(true)
            }
        }
    }
    
    fileprivate func toFeedbackMailCompose(_ closure:@escaping CompletedClosure) {
        ToastUtils.showActivity()
        zipArchive { [weak self] (state) in
            guard let strong = self else { return}
            ToastUtils.hideActivity()
            let result = strong.toFeedbackMailCompose()
            closure(result)
        }
    }
    
    
    fileprivate func toFeedbackMailCompose() -> Bool {
        if MFMailComposeViewController.canSendMail() {
            sendEmailAction();
            return true
        } else {
            ToastUtils.showToast(error: "Noemail_account.", to: Helper.currentVc?.view)
            return false
        }
    }
        
    fileprivate func sendEmailAction() {
        let email = "develop@shirajo.com";
        let mailCompose = MFMailComposeViewController();
        mailCompose.mailComposeDelegate = self;
        mailCompose.setToRecipients([email])
        mailCompose.setSubject("Feedback for \(name) (iOS)");
        /// 设置邮件的正文内容
        let emailContent = "Feedback content: \n\n\n";
        mailCompose.setMessageBody(emailContent, isHTML: false);
        /// Logger.zip 附件文件
        if let loggerPath = zipArchivePath() {
            if let data = NSData(contentsOfFile: loggerPath) {
                mailCompose.addAttachmentData(data as Data, mimeType: "zip", fileName: AppUtils.kLoggerZipName);
            }
        }
        Helper.currentVc?.present(mailCompose, animated: true, completion: nil);
    }
    
    internal func mailComposeController(_ controller: MFMailComposeViewController, didFinishWith result: MFMailComposeResult, error: Error?) {
        switch (result) {
        case .cancelled: UILog.v("Mail send canceled: 用户取消编辑")
        case .saved: UILog.v("Mail saved: 用户保存邮件")
        case .sent: UILog.v("Mail sent: 用户点击发送")
        case .failed: UILog.v("Mail send errored: \(error!.localizedDescription) : 用户尝试保存或发送邮件失败")
        @unknown default: break
        }
        controller.dismiss(animated: true, completion: nil);
    }
}

// MARK: - 将信息写入文件 Logger 目录下

extension AppUtils {
    
    fileprivate static var fileHandle: FileHandle? = nil

    enum LoggingOutputError:Error{
        case ErrorNoContent
        case ErrorNoDirectory
        case ErrorNoFile
        case ErrorFile
    }
    
    private func write<T>(with message:T, to fileName:String) {
        let content = "\(message)"
        let URL = FileManager.default.urls(for: .documentDirectory, in: .userDomainMask)[0]
        let directoryURL = URL.appendingPathComponent("Logger", isDirectory: true)
        appendText(directory: directoryURL, fileName: fileName, content: content)
    }
     
    //在文件末尾追加新内容
    private func appendText(directory url: URL, fileName name: String ,content: String) {
        do {
            try ensureValid(url, name, content)
        } catch let error as NSError {
            print("failed to append: \(error)")
        }
    }
    
    private func ensureValid(_ url: URL, _ name: String, _ content: String) throws {
        
        if  content.count == 0{
            throw LoggingOutputError.ErrorNoContent
        }
        
        var directory: ObjCBool = ObjCBool(false)
        let exists = FileManager.default.fileExists(atPath: url.path, isDirectory: &directory)
        if exists && directory.boolValue{
        } else {
            do {
                try FileManager.default.createDirectory(atPath: url.path, withIntermediateDirectories: true, attributes: nil)
            } catch {
                throw LoggingOutputError.ErrorNoDirectory
            }
        }
        
        let fileURL = url.appendingPathComponent("\(name).txt")
        
        if !FileManager.default.fileExists(atPath: fileURL.path) {
            if !FileManager.default.createFile(atPath: fileURL.path, contents: nil) {
                throw LoggingOutputError.ErrorNoFile
            } else {
                AppUtils.fileHandle?.closeFile()
                AppUtils.fileHandle = nil
            }
        }
        
        do {
            let stringToWrite = content + "\n"
            let data = stringToWrite.data(using: String.Encoding.utf8)!
            if AppUtils.fileHandle == nil { AppUtils.fileHandle = try FileHandle(forWritingTo: fileURL) }
            AppUtils.fileHandle?.truncateFile(atOffset:0)
            AppUtils.fileHandle?.synchronizeFile()
            AppUtils.fileHandle?.write(data)
        } catch {
            throw LoggingOutputError.ErrorFile
        }
    }
}

extension AppUtils {
    static var backgroundModes: [String] = Bundle.main.backgroundModes

    static func saveInfo() {
        UILog.d("\(shared.toJSONString(prettyPrint: true)!)")
        shared.write(with: shared.toJSONString(prettyPrint: true)!, to: "infomation")
    }
    
    static func toFeedbackMailCompose(_ closure:@escaping CompletedClosure) {
        shared.toFeedbackMailCompose(closure)
    }
}


// MARK: - Style for NavigationBar & TabBar

extension AppUtils {
    
    static func defaultStyle() {
        defaultNavigaionBarStyle()
        defaultTabBarStyle()
    }
    
    static func defaultNavigaionBarStyle() {
        let navigationBar = UINavigationBar.appearance()
        navigationBar.setBackgroundImage(UIImage(), for: .default)
        navigationBar.shadowImage = navBarShadowImage()
        
        navigationBar.barStyle = .black;
        navigationBar.isTranslucent = false;
        navigationBar.barTintColor = Colors.NavigationBar.barTint;
        navigationBar.tintColor = Colors.NavigationBar.tint;
        navigationBar.titleTextAttributes = [NSAttributedString.Key.foregroundColor: Colors.NavigationBar.foreground,
                                             NSAttributedString.Key.font: UIFont.boldCustomFont(ofSize: 18)]
        // let barButtonItem = UIBarButtonItem.appearance();
        // barButtonItem.tintColor = Colors.white;
    }

    static func navBarShadowImage() -> UIImage? {
        UIGraphicsBeginImageContextWithOptions(CGSize(width: UIDevice.scrWidth, height: 0.5), false, 0)
        let path = UIBezierPath.init(rect: CGRect.init(x: 0, y: 0, width: UIDevice.scrWidth, height: 0.5))
        /// 自定义NavigationBar分割线颜色
        Colors.NavigationBar.separator.setFill()
        path.fill()
        let image = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        return image
    }

    static func defaultTabBarStyle() {
        let tabBar = UITabBar.appearance()
        tabBar.backgroundImage = UIImage()
        tabBar.shadowImage = tabBarShadowImage()

        tabBar.barStyle = .default
        tabBar.isTranslucent = false;
        tabBar.barTintColor = Colors.TabBar.barTint;
    }

    static func tabBarShadowImage() -> UIImage? {
        UIGraphicsBeginImageContextWithOptions(CGSize(width: UIDevice.scrWidth, height: 0.5), false, 0)
        let path = UIBezierPath.init(rect: CGRect.init(x: 0, y: 0, width: UIDevice.scrWidth, height: 0.5))
        /// 自定义NavigationBar分割线颜色
        Colors.TabBar.separator.setFill()
        path.fill()
        let image = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        return image
    }
}

extension AppUtils {
    
    static let icon: UIImage  = Bundle.main.appIcon ?? UIImage()
    
    static func carrierName() -> String {
        let info = CTTelephonyNetworkInfo()
        if let carrier = info.subscriberCellularProvider, let name = carrier.carrierName {
            return name
        }
        return ""
    }
            
    static func deviceModel() -> String {
        
        var systemInfo = utsname()
        uname(&systemInfo)
            
        let machineMirror = Mirror(reflecting: systemInfo.machine)
        let identifier = machineMirror.children.reduce("") { identifier, element in
            guard let value = element.value as? Int8, value != 0 else { return identifier }
            return identifier + String(UnicodeScalar(UInt8(value)))
        }
        /// https://www.jianshu.com/p/0969c387ffe3
        switch identifier {
        case "i386", "x86_64": return "iPhone Simulator"
        case "iPhone1,1": return "iPhone"
        case "iPhone1,2": return "iPhone 3G"
        case "iPhone2,1": return "iPhone 3GS"
        case "iPhone3,1": return "iPhone 4"
        case "iPhone3,2": return "iPhone 4"
        case "iPhone3,3": return "iPhone 4"
        case "iPhone4,1": return "iPhone 4S"
        case "iPhone5,1": return "iPhone 5"
        case "iPhone5,2": return "iPhone 5"
        case "iPhone5,3": return "iPhone 5C"
        case "iPhone5,4": return "iPhone 5C"
        case "iPhone6,1": return "iPhone 5S"
        case "iPhone6,2": return "iPhone 5S"
        case "iPhone7,2": return "iPhone 6"
        case "iPhone7,1": return "iPhone 6 Plus"
        case "iPhone8,1": return "iPhone 6S"
        case "iPhone8,2": return "iPhone 6S Plus"
        case "iPhone8,4": return "iPhone SE"
        case "iPhone9,1": return "iPhone 7"
        case "iPhone9,3": return "iPhone 7"
        case "iPhone9,2": return "iPhone 7 Plus"
        case "iPhone9,4": return "iPhone 7 Plus"
        case "iPhone10,1": return "iPhone 8"
        case "iPhone10,4": return "iPhone 8"
        case "iPhone10,2": return "iPhone 8 Plus"
        case "iPhone10,5": return "iPhone 8 Plus"
        case "iPhone10,3": return "iPhone X"
        case "iPhone10,6": return "iPhone X"
        case "iPhone11,2": return "iPhone XS"
        case "iPhone11,4": return "iPhone XS Max"
        case "iPhone11,6": return "iPhone XS Max"
        case "iPhone11,8": return "iPhone XR"
        case "iPhone12,1": return "iPhone 11"
        case "iPhone12,3": return "iPhone 11 Pro"
        case "iPhone12,5": return "iPhone 11 Pro Max"
            
        case "iPad1,1": return "iPad 1"
        case "iPad2,1": return "iPad 2"
        case "iPad2,2": return "iPad 2"
        case "iPad2,3": return "iPad 2"
        case "iPad2,4": return "iPad 2"
        case "iPad3,1": return "iPad 3rd"
        case "iPad3,2": return "iPad 3rd"
        case "iPad3,3": return "iPad 3rd"
        case "iPad3,4": return "iPad 4th"
        case "iPad3,5": return "iPad 4th"
        case "iPad3,6": return "iPad 4th"
        case "iPad6,11": return "iPad 5th"
        case "iPad6,12": return "iPad 5th"
        case "iPad7,5": return "iPad 6th"
        case "iPad7,6": return "iPad 6th"
        case "iPad7,11": return "iPad 7th"
        case "iPad7,12": return "iPad 7th"
            
        case "iPad2,5": return "iPad Mini"
        case "iPad2,6": return "iPad Mini"
        case "iPad2,7": return "iPad Mini"
        case "iPad4,4": return "iPad Mini 2"
        case "iPad4,5": return "iPad Mini 2"
        case "iPad4,6": return "iPad Mini 2"
        case "iPad4,7": return "iPad Mini 3"
        case "iPad4,8": return "iPad Mini 3"
        case "iPad4,9": return "iPad Mini 3"
        case "iPad5,1": return "iPad Mini 4"
        case "iPad5,2": return "iPad Mini 4"
        case "iPad11,1": return "iPad Mini 5"
        case "iPad11,2": return "iPad Mini 5"
            
        case "iPad4,1": return "iPad Air"
        case "iPad4,2": return "iPad Air"
        case "iPad4,3": return "iPad Air"
        case "iPad5,3": return "iPad Air 2"
        case "iPad5,4": return "iPad Air 2"
        case "iPad11,3": return "iPad Air 3rd"
        case "iPad11,4": return "iPad Air 3rd"
            
        case "iPad6,7": return "iPad Pro 12.9-inch 1st"
        case "iPad6,8": return "iPad Pro 12.9-inch 1st"
        case "iPad6,3": return "iPad Pro 9.7-inch"
        case "iPad6,4": return "iPad Pro 9.7-inch"
        case "iPad7,1": return "iPad Pro 12.9-inch 2nd"
        case "iPad7,2": return "iPad Pro 12.9-inch 2nd"
        case "iPad7,3": return "iPad Pro 10.5-inch"
        case "iPad7,4": return "iPad Pro 10.5-inch"
        case "iPad8,5": return "iPad Pro 12.9-inch 3rd"
        case "iPad8,6": return "iPad Pro 12.9-inch 3rd"
        case "iPad8,7": return "iPad Pro 12.9-inch 3rd"
        case "iPad8,8": return "iPad Pro 12.9-inch 3rd"
        case "iPad8,1": return "iPad Pro 11-inch"
        case "iPad8,2": return "iPad Pro 11-inch"
        case "iPad8,3": return "iPad Pro 11-inch"
        case "iPad8,4": return "iPad Pro 11-inch"
            
        case "iPod1,1": return "iPod Touch 1st"
        case "iPod2,1": return "iPod Touch 2nd"
        case "iPod3,1": return "iPod Touch 3rd"
        case "iPod4,1": return "iPod Touch 4th"
        case "iPod5,1": return "iPod Touch 5th"
        case "iPod7,1": return "iPod Touch 6th"
            
        case "AppleTV2,1": return "Apple TV 2nd"
        case "AppleTV3,1": return "Apple TV 3rd"
        case "AppleTV3,2": return "Apple TV 3rd Rev A"
        case "AppleTV5,3": return "Apple TV HD"
        case "AppleTV6,2": return "Apple TV 4K"
        default: return identifier
        }
    }
}
