//
//  MarkerUtils.swift
//  CTFit
//
//  Created by Mac on 2020/6/4.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import Charts

struct MarkerUtils {
    
    class Step: BalloonMarker {
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let yValue = Int(entry.y);
            setLabel(String(format: "%d%@", yValue, SRString.Step.unit_step.locastr))
        }
    }

    class Distance: BalloonMarker {
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let yValue = entry.y;
            setLabel(String(format: "%0.2f%@", yValue, SRString.Unit.km.locastr))
        }
    }
    
    class ExerciseHeartRate: BalloonMarker {
        
        var startHour: Int = 0;
        var startMinute: Int = 0;
        private var startMinuteOfDay: Int { return startHour * 60 + startMinute; }
        
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let xAxis = Int(entry.x);
            let yAxis = Int(entry.y);
            
            let minuteOfDay = startMinuteOfDay + xAxis;
            let hourOfDay = minuteOfDay / 60;
            let minute = minuteOfDay % 60;
            
            let text = String(format: "%d %02d:%02d", yAxis, hourOfDay, minute);
            setLabel(text);
        }
    }
    
    
    class SleepDay: BalloonMarker {
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            if entry is SleepTypeBarEntry {
                let text = (entry as! SleepTypeBarEntry).getMarkerLabel();
                setLabel(text);
            }
        }
    }
    
    class SleepInterval: BalloonMarker {
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let yValue = entry.y;
            setLabel(String(format: "%0.1f%@", yValue, SRString.Unit.hours.locastr))
        }
    }

    
    class SleepHeart: BalloonMarker {
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let xAxis = Int(entry.x);
            let yAxis = Int(entry.y);
            
            let hourOfDay = (xAxis * 10 / 60 + 20) % 24; //  睡眠时间从前一天20:00开始
            let minute = (xAxis * 10) % 60;
            
            let text = String(format: "%d %02d:%02d", yAxis, hourOfDay, minute);
            setLabel(text);
        }
    }
    
    class HeartDay: BalloonMarker {
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let xAxis = Int(entry.x);
            let yAxis = Int(entry.y);
            
            let hourOfDay = xAxis * 10 / 60;
            let minute = (xAxis * 10) % 60;
            
            let text = String(format: "%d %02d:%02d", yAxis, hourOfDay, minute);
            setLabel(text);
        }
    }
    
    
}

class SleepBarSet: BarChartDataSet {
    override func color(atIndex index: Int) -> NSUIColor {
        guard let entry = self[index] as? SleepTypeBarEntry else { return super.color(atIndex: index) }
        return entry.color
    }
}

// MARK: - BarChartDataEntry & BarChartDataSet
class SleepTypeBarEntry: BarChartDataEntry {
    
    /// 睡眠状态类型
    var type: BDSleepType = .none
    
    /// 当前区间（连续相同状态区间）开始位置
    var startIndex: Int = 0
    
    
    static func create(x: Double, type: BDSleepType, start: Int) -> SleepTypeBarEntry {
        var y: Double = 0
        switch type {
        case .none: y = 0
        case .wake: y = 10
        case .deep: y = 15
        case .light: y = 20
        }
        let entry = SleepTypeBarEntry(x: x, y: y)
        entry.type = type
        entry.startIndex = start
        return entry
    }
    
    func getMarkerLabel() -> String {
        let hourUnit =  SRString.Unit.hours.locastr
        let endMinute = Int(x)
        let startTime = timeFromMinutes(startIndex)
        let endTime = timeFromMinutes(endMinute)
        let duration = Float(endMinute - startIndex) / 60
        return String(format: "%@%.1f%@\n(%02d:%02d-%02d:%02d)", text, duration, hourUnit,
                          startTime.hour, startTime.minute, endTime.hour, endTime.minute)
    }
    
    fileprivate func timeFromMinutes(_ minutes: Int) -> (hour:Int, minute:Int) {
        let hour = (minutes / 60 + 20) % 24
        let minute = minutes % 60
        return (hour, minute)
    }

    var text: String {
        switch type {
        case .none, .wake:  return SRString.Sleep.wake.locastr
        case .light:        return SRString.Sleep.light_sleep.locastr
        case .deep:         return SRString.Sleep.deep_sleep.locastr
        }
    }
    
    var color: UIColor {
        switch type {
        case .none, .wake:  return Colors.Sleep.wake
        case .light:        return Colors.Sleep.light
        case .deep:         return Colors.Sleep.deep
        }
    }
}





extension MarkerUtils {
    
    class Detect: BalloonMarker {
        
        var type :BDDetectType!
        
        var items :[BDDetect]?
        
        convenience init(type: BDDetectType, items :[BDDetect]?) {
            self.init()
            self.type = type
            self.items = items
        }
        
        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let xAxis = Int(entry.x);
            let yAxis = Int(entry.y);
            var text = "\(yAxis)"
            
            if type == .bp, let hasItems = items {
                let item :BDDetect = hasItems[xAxis]
                text =  "\(item.heartorsbp)/\(item.spo2ordbp)";
            }
            text = String(format:"%@ %@", text, type.unit)
            setLabel(text);
        }
        
    }
    
    class SportSpeed: BalloonMarker {
        var items :[SportPoint]?
        
        convenience init(items :[SportPoint]?) {
            self.init()
            self.items = items
        }

        override func refreshContent(entry: ChartDataEntry, highlight: Highlight) {
            let xAxis = Int(entry.x);
            let speed = entry.y
            var text = String(format: "%.2f", speed);
            if let hasItems = items {
                let item :SportPoint = hasItems[xAxis]
                text = String(format:"%@ %@\n%@", text, speedUnit(), timeStr(timeInterval: TimeInterval(item.time)))
            }
            setLabel(text);
        }
        
        private func speedUnit() -> String {
            return BandSettings.isImperial ? SRString.Unit.mile_ph.locastr : SRString.Unit.km_ph.locastr
        }
        
        let timeFormatter :DateFormatter = {
            let fm = DateFormatter();
            fm.dateFormat = "HH:mm:ss"
            return fm;
        }()
        
    
        func timeStr(timeInterval: TimeInterval) -> String {
            timeFormatter.string(from: Date(timeIntervalSince1970: timeInterval))
        }
    }
}

