//
//  PersonalDataViewController.swift
//  CTFit
//
//  Created by Mac on 2020/12/15.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import TableViewManager
import RxSwift
import Toast

class PersonalDataViewController: BaseViewController, CustomView {

    typealias ViewClass = BaseTableView
    
    override init() {
        super.init()
        hidesBottomBarWhenPushed = true
        navigationItem.title = SRString.Application.personal_data.locastr
    }

    override func loadView() {
        super.loadView()
        view = ViewClass()
    }
    
    override func initNavigationBar() {
        super.initNavigationBar()
        navigationItem.rightBarButtonItem = exportBarItem()
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        addSubviews()
        update()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        customView.refreshTableView()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        super.viewDidDisappear(animated)
    }
    
    private func update() {
        let items = CSVHelper.fileList().map { PersonalDataItem(with: $0) }
        section.items.removeAll()
        items.forEach { (item) in section.add(item: item) }
        tableViewManger.reload()
        nodataLabel.isHidden = section.items.count > 0
        items.forEach { (item) in
            item.setDeletionHandler { [weak self] (t) in
                guard let strong = self,
                      let theItem = t as? PersonalDataItem else { return }
                strong.onDelete(for: theItem)
            }
        }
    }
    
    private func onDelete(for item: PersonalDataItem) {
        CSVHelper.deleteFile(item.fileURL)
        item.delete()
        nodataLabel.isHidden = section.items.count > 0
    }
    
    // MARK: - TableViewManager
        
    fileprivate lazy var tableViewManger: ZJTableViewManager = {
        let manager = ZJTableViewManager(tableView: customView.tableView)
        manager.register(PersonalDataCell.self, PersonalDataItem.self)
        manager.add(section: section);
        return manager;
    } ()
    
    fileprivate lazy var section: BaseTableViewSection = {
        let section = BaseTableViewSection()
        section.headerHeight = 0.01
        section.footerHeight = 0.01
        return section
    } ()
    
    // MARK: - View customer
    private func addSubviews() {
        [nodataLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            self.view.addSubview(view)
        }
        
        nodataLabel.snp.makeConstraints { (make) in
            make.center.equalTo(customView.tableView.snp.center)
        }
    }
    
    private lazy var nodataLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.gray
        label.text = SRString.Application.no_file.locastr
        label.isHidden = false
        return label
    }()
}

extension PersonalDataViewController {
    @objc private func exportBarItem() -> UIBarButtonItem? { /// 导出
        let item = UIBarButtonItem(title: SRString.Application.export.locastr, style: .plain, target: self, action: #selector(onExport));
        return item;
    }
       
    @objc private func onExport() {
        ToastUtils.showToastCCV(normal: SRString.Application.exporting.locastr)
        CSVHelper.export { [weak self] in
            guard let strong = self else { return }
            ToastUtils.showToastCCV(success: SRString.Application.export_success.locastr)
            strong.update()
        }
    }
}
