//
//  ApplicationItem.swift
//  CTFit
//
//  Created by Mac on 2020/9/30.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import UIKit
import RxSwift
import TableViewManager

enum ApplicationItemType {
    case apple_health, personal_data, personal_data_delete, feedback, about
    var text: String {
        switch self {
        case .apple_health: return SRString.Application.apple_health.locastr
        case .personal_data: return SRString.Application.personal_data.locastr
        case .personal_data_delete: return SRString.Application.personal_data_delete.locastr
        case .feedback: return SRString.Application.feedback.locastr
        case .about: return SRString.Application.about.locastr
        }
    }
    
    static let collection: [ApplicationItemType] = [.apple_health, .personal_data, .personal_data_delete, .feedback, .about]
}

final class ApplicationItem: BaseTableViewItem {
    
    // MARK: - Cell source
    var text: String? { type.text }
    
    // MARK: - Initialize
    override init() {
        super.init()
        cellHeight = 60
        selectionStyle = .default
        accessoryType = .none
    }
    
    convenience init(_ type: ApplicationItemType) {
        self.init()
        self.type = type
        onHandler()
    }
    
    // MARK: - Field
    var type: ApplicationItemType = .about
    private let disposeBag = DisposeBag()
}

import FileBrowser
extension ApplicationItem {
    private func onHandler(){
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler()
        }
    }
    
    private func didSelectedHandler() {
        switch type {
        case .apple_health: push(AppleHealthViewController())
        case .personal_data: push(PersonalDataViewController())
        case .personal_data_delete: clearUserData()
        case .feedback: AppUtils.toFeedbackMailCompose { (_) in }
        case .about: push(AboutViewController())
        }
    }
    
    private func push(_ vc: BaseViewController) {
        Helper.currentVc?.navigationController?.pushViewController(vc, animated: true)
    }
    
    private func clearUserData() {
        AlertManager.show(title: nil, message: SRString.Application.personal_data_delete_alert.locastr, okHander: { (_) in
            BandData.deletePersonalData()
            ToastUtils.showToastCCV(normal: SRString.Application.personal_data_deleted.locastr)
        }, cancelHander: { (_) in })
    }
}
