//
//  VibrateSelectItem.swift
//  CTFit
//
//  Created by Mac on 2020/7/16.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import TableViewManager

enum VibrateSelectType {
    case bell, bellCount
    var text: String {
        switch self {
        case .bell: return SRString.Vibrate.vibrate.locastr
        case .bellCount: return SRString.Vibrate.repeat.locastr
        }
    }
}

final class VibrateSelectItem: BaseExpandTreeCellItem {
    
    func setValueChanged(_ closure: @escaping ValueChangedClosure) {
        self.valueChangedClosure = closure
    }

    // MARK: - Cell source
    var text: String { type.text }
    var detailText: String {
        switch type {
        case .bell: return vibrate.type.text
        case .bellCount:
            if vibrate.count == 0 { return SRString.Clock.repeat_none.locastr }
            return "\(vibrate.count)"
        }
    }
    
    // MARK: - Initialize
    convenience init(with type: VibrateSelectType, _ vibrate: BDVibrate) {
        self.init()
        self.type = type
        self.vibrate = vibrate
        // onHander()
    }
    
    
    // MARK: - private field
    private var type: VibrateSelectType = .bell
    private var vibrate: BDVibrate = BDVibrate()
    private lazy var dialog: Dialog = Dialog(with: .bottom)

    typealias ValueChangedClosure = (_ vibrate: BDVibrate) -> Void
    private var valueChangedClosure: ValueChangedClosure? = nil
    private func onValueChanged() { if let closure = valueChangedClosure { closure(vibrate) } }

    
    // MARK: - Sub picker item
    
    func addSubPickerItem(to section: ZJTableViewSection) {
        addSub(item: pickerItem, section: section)
    }
    
    private lazy var pickerItem: PickerItem = {
        let item = PickerItem(components: components, selectRows: selectRows)
        item.setItemSelectedClosure { [weak self] (text, component) in
            guard let strong = self else { return }
            strong.onIemSelected(text, component)
        }
        return item
    } ()
    
    private lazy var components: [[String]] = {
        switch type {
        case .bell: return PickerOptionsUitls.components(.bell)
        case .bellCount: return PickerOptionsUitls.components(.bellCount)
        }
    } ()
    
    private lazy var selectRows: [String] = {
        switch type {
        case .bell: return [vibrate.type.text]
        case .bellCount: return ["\(vibrate.count)"]
        }
    } ()
    
    private func onIemSelected(_ text: String, _ component: Int) {
        switch type {
        case .bell:
            vibrate.type = BDVibrateType.parse(text)
        case .bellCount:
            let value: UInt8 = UInt8(text)!
            vibrate.count = value
        }
        reload()
    }
}

           
extension VibrateSelectItem {
    
    private func onHander() {
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.pickerUser()
        }
    }
    
    private func pickerUser() {

        let anVibrate = vibrate.copy() 

        let leftTap: DialogPickerView.ButtonTapEvent = { [weak self] (sender) in
            guard let strong = self else { return }
            strong.dialog.dismiss(true) { _ in }
        }
        
        let rightTap: DialogPickerView.ButtonTapEvent = { [weak self] (sender) in
            guard let strong = self else { return }
            strong.vibrate = anVibrate.copy() 
            strong.reload()
            strong.dialog.dismiss(true) { _ in }
            strong.onValueChanged()
        }
        
        
        if type == .bell {
            let pickerView = DialogPickerView(title: type.text,
                                              components: PickerOptionsUitls.components(.bell),
                                              leftButtonTitle: SRString.Base.cancel.locastr,
                                              leftButtonTap: leftTap,
                                              rightButtonTitle: SRString.Base.ok.locastr,
                                              rightButtonTap: rightTap)
            pickerView.select([anVibrate.type.text], false)
            pickerView.setItemSelected { (rowItem, component) in
                anVibrate.type = BDVibrateType.parse(rowItem)
            }
            
            dialog.pop(pickerView, completed: { _ in })
            
        }
        
        if type == .bellCount {
            let pickerView = DialogPickerView(title: type.text,
                                              components: PickerOptionsUitls.components(.bellCount),
                                              leftButtonTitle: SRString.Base.cancel.locastr,
                                              leftButtonTap: leftTap,
                                              rightButtonTitle: SRString.Base.ok.locastr,
                                              rightButtonTap: rightTap)
            pickerView.select(["\(anVibrate.count)"], false)
            pickerView.setItemSelected { (rowItem, component) in
                let value: UInt8 = UInt8(rowItem)!
                anVibrate.count = value
            }
            dialog.pop(pickerView, completed: { _ in } ) 
        }
    }
}

