//
//  FaqViewController.swift
//  CTFit
//
//  Created by Mac on 2020/9/18.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import WebKit

class FaqViewController: BaseViewController {
    
    override init() {
        super.init()
        hidesBottomBarWhenPushed = true
        modalPresentationStyle = .fullScreen
        navigationItem.title = SRString.Settings.faq.locastr
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setSubviews()
        adapterDark()
        loadFAQ()
    }
    
    private func setSubviews() {
        [webView].forEach { v in
            v.translatesAutoresizingMaskIntoConstraints = false
            view.addSubview(v)
        }
        webView.snp.makeConstraints { (make) in
            make.leading.trailing.equalToSuperview();
            make.top.equalTo(self.topLayoutGuide.snp.bottom)
            make.bottom.equalTo(self.bottomLayoutGuide.snp.top)
        }
    }
    
    private func loadFAQB() {
        DispatchQueue.global().async {
            guard let htmlString = try? String(contentsOf: self.faqUrl, encoding: .utf8) else { return }
            let newHtmlString = htmlString.replacingOccurrences(of: "CTFIT", with: AppUtils.shared.name);
            DispatchQueue.main.async { [unowned self] in
                self.webView.loadHTMLString(newHtmlString, baseURL: nil)
            }
        }
    }
    
    private func loadFAQ() {
        self.webView.loadFileURL(faqUrl, allowingReadAccessTo: faqUrl)
    }

    // MARK: - View customer
    
    private lazy var faqFile: String  = {
        switch NELanguage.type() {
        case .en:       return "faq_setting_en.html"
        case .zh_Hans, .zh_Hant:  return "faq_setting_zh.html"
        case .ja:       return "faq_setting_jp.html"
        case .de:       return "faq_setting_de.html"
        case .fr:       return "faq_setting_fr.html"
        case .it:       return "faq_setting_it.html"
        case .es:       return "faq_setting_es.html"
        case .pt:       return "faq_setting_pt.html"
        default:        return "faq_setting_en.html"
        }
    } ()
    
    private lazy var faqUrl: URL  = {
        var url = Bundle.main.bundleURL
        url.appendPathComponent(faqFile)
        return url
    } ()
    
    private lazy var urlRequest: URLRequest = URLRequest(url: faqUrl)

    private lazy var userScript: WKUserScript = {
        let javascript = "document.documentElement.style.webkitTouchCallout='none';" + "document.documentElement.style.webkitUserSelect='none';"
        let script = WKUserScript(source: javascript, injectionTime: .atDocumentStart, forMainFrameOnly: false)
        return script
    } ()
    
    private lazy var contentController: WKUserContentController = {
        let controller = WKUserContentController()
        controller.addUserScript(userScript)
        return controller
    } ()
    
    private lazy var configuration: WKWebViewConfiguration = {
        let config = WKWebViewConfiguration()
        config.userContentController = contentController
        return config
    } ()
    
    private lazy var webView: WKWebView = {
        let view = WKWebView(frame: CGRect.zero, configuration: configuration)
        view.scrollView.backgroundColor = Colors.Base.background
        view.isHidden = true
        view.allowsLinkPreview = true;
        view.navigationDelegate = self
        return view
    } ()
}

extension FaqViewController: WKNavigationDelegate {
    
    func webView(_ webView: WKWebView, didFinish navigation: WKNavigation!) {
        adapterDark()
    }
    
    override func traitCollectionDidChange(_ previousTraitCollection: UITraitCollection?) {
        super.traitCollectionDidChange(previousTraitCollection)
        /// 每次模式改变的时候, 这里都会执行
        adapterDark()
    }
    
    private func adapterDark() {
        var backgroundColor = "'#FFFFFF'"
        var labelColor = "'#000000'"
        if #available(iOS 13.0, *),
            traitCollection.userInterfaceStyle == .dark {
            backgroundColor = "'#000000'"
            labelColor = "'#FFFFFF'"
        }
        self.webView.evaluateJavaScript("document.body.style.backgroundColor=\(backgroundColor)", completionHandler: nil)
        for i in 0...6 {
            let js1 = "document.getElementsByTagName('summary')[\(i)].style.color=\(labelColor)"
            let js2 = "document.getElementsByTagName('summary::-webkit-details-marker')[\(i)].style.color=\(labelColor)"
            self.webView.evaluateJavaScript(js1, completionHandler: nil)
            self.webView.evaluateJavaScript(js2, completionHandler: nil)
        }
        DispatchQueue.main.asyncAfter(deadline: .now()+0.2) {
            let transition = CATransition()
            transition.duration = 0.2
            self.webView.isHidden = false
            self.webView.layer.add(transition, forKey: nil)
        }
    }
}
