//
//  HomeCircleView.swift
//  CTFit
//
//  Created by Mac on 2020/9/3.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import RxSwift

class HomeCircleView: UIView {
    func viewWillAppear() {
        onUpdate()
    }
    
    func viewDidAppear() {
        if circleMenu.currentItem != 0 {
            DispatchQueue.main.asyncAfter(deadline: .now()+0.05) {
                self.circleMenu.animateToSelectedItem(selected: 0);
            }
        }
    }
    
    func onDidSelectedMenu(_ closure: @escaping DidSelectedMenu) {
        didSelectedMenu = closure
    }
    
    func onDidSelectedAttr(_ closure: @escaping DidSelectedAttr) {
        didSelectedAttr = closure
    }
    
    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func traitCollectionDidChange(_ previousTraitCollection: UITraitCollection?) {
        super.traitCollectionDidChange(previousTraitCollection)
        backgroudView.image = bgImage
    }
    
    init() {
        super.init(frame: .zero)
        addSubviews()
        onUpdate()
        observers()
        addLongPressGesture()
    }
    
    func onUpdate() {
        BandData.queryHome(with: Date()) { [weak self] total in
            guard let strong = self else { return }
            strong.onUpdate(total ?? BDTotal())
        }
    }
    
    private func onUpdate(_ total: BDTotal) {
        let goalStep = BandSettings.goal.step
        let progress: CGFloat = CGFloat(total.steps)/CGFloat(goalStep)
        stepProgressBar.value = "\(total.steps)"
        stepProgressBar.setProgress(progress, animated: true)
        bottomView.onUpdate(total: total)
    }
    
    
    private func observers()  {
        BandData.updatedOutput.observeOn(MainScheduler.instance).subscribe(onNext: { [weak self] (type) in
            guard let strong = self else { return }

            switch type {
            case .record(let type, let dateStr):
                guard dateStr == CalendarUtils.dateStr(date: Date()) else { return }
                guard type == .total || type == .heart  else { return }
                strong.onUpdate()
            default : break
            }
        }).disposed(by: disposeBag)
        
        BandSettings.observable.subscribe(onNext: { [weak self] (options) in
            guard let strong = self else { return }
            guard options.contains(.system) || options.contains(.goal) else { return }
            strong.onUpdate()
        }).disposed(by: disposeBag)
    }
    
    // MARK: - View customer

    private func addSubviews() {
        let items = [backgroudView, circleMenu, bottomView]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        backgroudView.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
        
        circleMenu.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview();
            make.centerY.equalToSuperview().offset(-85)
        }
        
        bottomView.snp.makeConstraints { (make) in
            make.left.right.equalTo(0);
            make.height.equalTo(100);
            make.bottom.equalTo(-Constraints.normal-UIDevice.bottomLayoutGuide);
        }
    }
    
    var bgImage: UIImage {
        ImageRepo.HomeCircle.bacground
    }
    
    private lazy var backgroudView: UIImageView = {
        let view = UIImageView(image: bgImage)
        view.backgroundColor = Colors.white
        view.contentMode = .scaleAspectFill
        view.clipsToBounds = true
        return view
    } ()
    
    private lazy var circleMenu: CircleMenu = {
        let memu = CircleMenu();
        memu.onDidSelectedMenu { [weak self] (type, repeater) in
            guard let strong = self else { return}
            strong.didSelectedMenu?(type)
        }
        var items = [CircleMenuItem]();
        for type in CircleMenuItemType.collection { items.append(CircleMenuItem(type)) }
        memu.items = items;
       return memu
    } ()
    
    private lazy var bottomView: HomeCircleBottomStack = {
        let view = HomeCircleBottomStack()
        view.onDidSelectedAttr {[weak self] (type) in
            guard let strong = self else { return}
            strong.didSelectedAttr?(type)
        }
        return view
    } ()
    
    private lazy var stepProgressBar: StepProgressBar = circleMenu.centerView
    typealias DidSelectedMenu = (_ type: CircleMenuItemType) -> Void
    typealias DidSelectedAttr = (_ type: HomeCircleAttrType) -> Void
    private var didSelectedMenu: DidSelectedMenu?
    private var didSelectedAttr: DidSelectedAttr?
    private lazy var disposeBag = DisposeBag()
}


// MARK: - longPressGesture
extension HomeCircleView {
    private func addLongPressGesture() {
        let gesture = UILongPressGestureRecognizer(target: self, action: #selector(longPressGestureAction(_:)))
        // 设定最小的长按时间 按不够这个时间不响应手势
        gesture.minimumPressDuration = 1.5;
        // 添加到整个View
        self.addGestureRecognizer(gesture)
    }
    
    @objc
    private func longPressGestureAction(_ gesture: UILongPressGestureRecognizer) {
        if gesture.state == .began {
            onLongPressAction()
        }
    }
    
    private func onLongPressAction() {
        Helper.currentVc?.navigationController?.pushViewController(ProtocolViewController(), animated: true)
    }
}


// MARK: - HomeCircleBottomStack

class HomeCircleBottomStack: UIStackView {
    private lazy var disposeBag = DisposeBag()
    typealias DidSelectedAttr = (_ type: HomeCircleAttrType) -> Void
    private var didSelectedAttr: DidSelectedAttr?
    
    func onUpdate(total: BDTotal) {
        updateDistance(total)
        stepAttr.value = "\(total.steps)"
        calorieAttr.value = "\(Int(total.calories/1000))"
        heartAttr.value = "\(total.latestHeart)"
    }
    
    func updateDistance(_ total: BDTotal) {
        if !BandSettings.isImperial {
            distanceAttr.value = String(format: "%.2f", BandCalculator.km(device: BDCalculatorType(total.distance)))
            distanceAttr.unit = SRString.Unit.km.locastr
        } else {
            distanceAttr.value = String(format: "%.2f", BandCalculator.mile(device: BDCalculatorType(total.distance)))
            distanceAttr.unit = SRString.Unit.mile.locastr
        }
    }
    
    func onDidSelectedAttr(_ closure: @escaping DidSelectedAttr) {
        didSelectedAttr = closure
    }
    
    // MARK: Initialize
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize();
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    private func initialize() {
        axis = .horizontal
        distribution = .fillEqually
        self.addArrangedSubview(stepAttr)
        self.addArrangedSubview(distanceAttr)
        self.addArrangedSubview(calorieAttr)
        self.addArrangedSubview(heartAttr)
    }
    
    @objc private func onTapAction(sender: HomeCircleAttr) {
        didSelectedAttr?(sender.type)
    }
    
    // MARK: View customer

    private lazy var stepAttr: HomeCircleAttr = {
        let attr = HomeCircleAttr(.step);
        attr.image = ImageRepo.HomeCircle.steps;
        attr.value = "0";
        attr.unit = SRString.Step.unit_step.locastr
        attr.addTarget(self, action: #selector(onTapAction(sender:)), for: .touchUpInside)
        return attr;
    }()
    private lazy var distanceAttr: HomeCircleAttr = {
        let attr = HomeCircleAttr(.distance);
        attr.image = ImageRepo.HomeCircle.distance;
        attr.value = "0";
        attr.unit = SRString.Unit.km.locastr
        attr.addTarget(self, action: #selector(onTapAction(sender:)), for: .touchUpInside)
        return attr;
    }()
    private lazy var calorieAttr: HomeCircleAttr = {
        let attr = HomeCircleAttr(.calorie);
        attr.image = ImageRepo.HomeCircle.calorie;
        attr.value = "0";
        attr.unit = SRString.Step.unit_calories.locastr
        attr.addTarget(self, action: #selector(onTapAction(sender:)), for: .touchUpInside)
        return attr;
    }()
    private lazy var heartAttr: HomeCircleAttr = {
        let attr = HomeCircleAttr(.heart);
        attr.image = ImageRepo.HomeCircle.heartrate;
        attr.value = "0";
        attr.unit = SRString.Heart.bpm.locastr
        attr.addTarget(self, action: #selector(onTapAction(sender:)), for: .touchUpInside)
        return attr;
    }()
}

// MARK: - HomeCircleAttr
enum HomeCircleAttrType {
    case step, distance, calorie, heart
}

class HomeCircleAttr: UIControl {
    
    // MARK: - Properties
    var image :UIImage? {
        didSet { imageView.image = image; }
    }
    var value :String? {
        didSet { valueLabel.text = value; }
    }
    var unit :String? {
        didSet { unitLabel.text = unit; }
    }
    
    var type: HomeCircleAttrType = .step

    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(_ type: HomeCircleAttrType) {
        super.init(frame: .zero)
        self.type = type
        initialize()
    }

    override func touchesEnded(_ touches: Set<UITouch>, with event: UIEvent?) {
        let touch = touches.first;
        if let hasTouch = touch {
            let location = hasTouch.location(in: self);
            if self.bounds.contains(location) {
                sendActions(for: .touchUpInside);
            }
        }
    }
    
    private func initialize() {
        [imageView, valueLabel, unitLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        imageView.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview();
            make.size.equalTo(24)
            make.top.equalTo(8)
        }
        valueLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview();
            make.top.equalTo(imageView.snp.bottom).offset(16)
        }
        unitLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview();
            make.top.equalTo(valueLabel.snp.bottom).offset(8)
        }
    }
    
    private lazy var imageView: UIImageView = {
        let view = UIImageView();
        view.contentMode = .scaleAspectFit
        return view
    } ()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.textColor = Colors.white
        return label
    } ()
    
    private lazy var unitLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.font = .customFont(ofSize: 14)
        label.textColor = Colors.white
        return label
    } ()
    
}
